/*
 * $QNXLicenseC:
 * Copyright 2010, QNX Software Systems. All Rights Reserved.
 *
 * You must obtain a written license from and pay applicable
 * license fees to QNX Software Systems before you may reproduce,
 * modify or distribute this software, or any work that includes
 * all or part of this software.   Free development licenses are
 * available for evaluation and non-commercial purposes.  For more
 * information visit http://licensing.qnx.com or email
 * licensing@qnx.com.
 *
 * This file may contain contributions from others.  Please review
 * this entire file for other proprietary rights or license notices,
 * as well as the QNX Development Suite License Guide at
 * http://licensing.qnx.com/license-guide/ for other information.
 * $
 */

#include <stddef.h>
#include <string.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <malloc.h>

#include "touch.h"
#include "save_cm_cp.h"

#ifndef UNITTEST
#include "input/mtouch_log.h"
#include "input/mtouch_driver.h"
#include "input/event_types.h"
#include "input/inputtrace.h"
#include "input/parseopts.h"
#include <devctl.h>
#include <hw/i2c.h>
#include <sys/neutrino.h>
#include <sys/slog.h>
#include <gulliver.h>
#include "utils.h"
#include "pal_boot.h"
#else
#include "err_mem_header.h"
#include "pthread_header.h"
#include "input_utest_header.h"
#include "ipc_header.h"
#include "timer_header.h"
#endif

#include <stdarg.h>
#include <string.h>

#define RESET_ALL(fail_label, error) \
    if (!cypress->debug_raw_mode) { \
        rc = reset_all_loop(cypress, error); \
        if (rc < 0) { \
            goto fail_label; \
        } \
    }
#define RC_CHECK_RESET_ALL(fail_label) \
    if (rc < 0) { \
        RESET_ALL(fail_label, 1); \
    }

volatile _Uint8t   xferbuf[CYPRESS_I2C_DATA_READ_LENGTH];

int cypress_platform_init(struct cypress_dev* dev);
void cypress_platform_fini(struct cypress_dev* dev);
uint16_t cypress_extract_data(cypress_dev_t* cypress, uint8_t *buf, uint8_t byte_offset, uint8_t bit_offset, uint8_t num_bits);


/* Write critical logs into errmem */
void error_memory(const char * fmt, ...)
{
#define MAXLINE    1024  /* max text line length */

    va_list ap;
    char buf[MAXLINE] = {0};

    va_start(ap, fmt);
    vsnprintf(buf, sizeof(buf), fmt, ap);
    va_end(ap);

    vWritePrintfErrmem(buf);

}

int reset_all(cypress_dev_t* cypress, int error)
{
    return 0;
}

int
safe_usleep(useconds_t usec)
{
    struct timespec ts;
    int rc;

    ts.tv_sec = usec / 1000000;
    ts.tv_nsec = (usec % 1000000) * 1000;
    while (1) {
        rc = nanosleep(&ts, &ts);
        if (EINTR != rc) {
            break;
        }
    }

    return rc;
}

int
cypress_set_timer(cypress_dev_t* cypress, unsigned sec, unsigned nsec, timer_t timerid)
{
    int rc = 0;
    struct itimerspec itime;

    itime.it_value.tv_sec = sec;
    itime.it_value.tv_nsec = nsec;
    itime.it_interval.tv_sec = 0;
    itime.it_interval.tv_nsec = 0;

    rc = timer_settime(timerid, 0, &itime, NULL);

    if (EOK != rc) {
        mtouch_error (cypress->log_name, "Failed to set the heartbeat timer to %u sec %u nsec.  Error: %d", sec, nsec, rc);
        error_memory("Cypress_Touch: Failed to set the heartbeat timer to %u sec %u nsec.  Error: %d", sec, nsec, rc);
        return -1;
    }

    return EOK;
}

int
get_contact_id(void* packet, uint8_t digit_idx, uint32_t* contact_id, void* arg)
{
    *contact_id = digit_idx;

    return EOK;
}

int
is_contact_down(void* packet, uint8_t digit_idx, int* valid, void* arg)
{
    cypress_dev_t* cypress =(cypress_dev_t *) arg;
    touch_report_t *touch_packet =(touch_report_t *) packet;

    if (packet == NULL || valid == NULL || arg == NULL) {
        mtouch_error (cypress->log_name, "%s NULL pointer", __FUNCTION__);
        error_memory ("Cypress_Touch: %s NULL pointer", __FUNCTION__);
        return -1;
    }

    *valid = touch_packet[digit_idx].status;

    /* (Re)Start timer, need to guarantee that it won't miss the release event  */
    if ((cypress->release_timeout) && (*valid)) {
        cypress_set_timer(cypress, 0, cypress->release_timeout, cypress->inject_release_timerid[digit_idx]);
    } else if (cypress->release_timeout) {
        cypress_set_timer(cypress, 0, 0, cypress->inject_release_timerid[digit_idx]);
    }

    return EOK;
}

int
get_coords(void* packet, uint8_t digit_idx, int32_t* x, int32_t* y, void* arg)
{
    cypress_dev_t* cypress =(cypress_dev_t *) arg;
    touch_report_t *touch_packet =(touch_report_t *) packet;

    *x = touch_packet[digit_idx].x;
    *y = touch_packet[digit_idx].y;

    if(cypress->invert_x)
        *x = cypress->width - 1 - *x;
    if(cypress->invert_y)
        *y = cypress->height - 1 - *y;

    if (cypress->verbose > 7)
        mtouch_info (cypress->log_name, "Touch Coordinates sent to screen: Finger %d x: %d, y: %d", digit_idx, *x, *y);

    cypress->last_coords.x = *x;
    cypress->last_coords.y = *y;

    return EOK;
}

void
get_seq_id(void* packet, uint32_t* seq_id, void* arg)
{
    cypress_dev_t* cypress =(cypress_dev_t *) arg;

    *seq_id = cypress->seq_id++;
    cypress->seq_id &= ~(INPUTTRACE_SEQ_TYPE(INPUTTRACE_SEQ_TYPE_MASK));
}

int get_touch_width(void* packet, _Uint8t digit_idx, _Uint32t* touch_width, void* arg)
{
    cypress_dev_t* cypress =(cypress_dev_t *) arg;
    touch_report_t *touch_packet =(touch_report_t *) packet;

    *touch_width = touch_packet[digit_idx].width;

    if (cypress->verbose > 7)
        mtouch_info (cypress->log_name, "Touch width sent to screen: Finger %d width: %d", digit_idx, *touch_width);

    return EOK;
}

int get_touch_height(void* packet, _Uint8t digit_idx, _Uint32t* touch_height, void* arg)
{
    cypress_dev_t* cypress =(cypress_dev_t *) arg;
    touch_report_t *touch_packet =(touch_report_t *) packet;

    *touch_height = touch_packet[digit_idx].height;

    if (cypress->verbose > 7)
        mtouch_info (cypress->log_name, "Touch height sent to screen: Finger %d height: %d", digit_idx, *touch_height);

    return EOK;
}

int get_touch_pressure(void* packet, _Uint8t digit_idx, _Uint32t* touch_pressure, void* arg)
{
    cypress_dev_t* cypress =(cypress_dev_t *) arg;
    touch_report_t *touch_packet =(touch_report_t *) packet;

    *touch_pressure = touch_packet[digit_idx].pressure;

    if (cypress->verbose > 7)
        mtouch_info (cypress->log_name, "Touch pressure sent to screen: Finger %d pressure: %d", digit_idx, *touch_pressure);
    return EOK;
}

int get_touch_orientation(void* packet, _Uint8t digit_idx, _Uint32t* touch_orientation, void* arg)
{
    cypress_dev_t* cypress =(cypress_dev_t *) arg;
    touch_report_t *touch_packet =(touch_report_t *) packet;

    *touch_orientation = touch_packet[digit_idx].orientation;

    if (cypress->verbose > 7)
        mtouch_info (cypress->log_name, "Touch orientation sent to screen: Finger %d orientation: %d", digit_idx, *touch_orientation);
    return EOK;
}

int
dump_i2c_packet(cypress_dev_t* cypress, uint8_t addr, uint8_t *buf, int len)
{
    int i;
    mtouch_info (cypress->log_name, "Writing to Addr %d length %d data:", addr, len);

    for (i = 0; i < len; i++) {
        mtouch_info (cypress->log_name, "%x", buf[i]);
    }

    return EOK;
}

int
cypress_set_heartbeat_timer(cypress_dev_t* cypress, unsigned sec)
{
    int rc;

    cypress->itime.it_value.tv_sec = sec;
    cypress->itime.it_value.tv_nsec = 0;
    cypress->itime.it_interval.tv_sec = 0;
    cypress->itime.it_interval.tv_nsec = 0;

    /* Set heartbeat timer */
    rc = timer_settime(cypress->heartbeat_timerid, 0, &(cypress->itime), NULL);

    if (EOK != rc) {
        mtouch_error (cypress->log_name, "Failed to set the heartbeat timer to %d.  Error: %d", sec, rc);
        error_memory ("Cypress_Touch: Failed to set the heartbeat timer to %d.  Error: %d", sec, rc);
        return -1;
    }

    return EOK;
}

int cypress_set_mode_change_timer(cypress_dev_t* cypress, unsigned nsec)
{
    int rc;

    cypress->itime.it_value.tv_sec = 0;
    cypress->itime.it_value.tv_nsec = nsec;
    cypress->itime.it_interval.tv_sec = 0;
    cypress->itime.it_interval.tv_nsec = 0;

    /* Set heartbeat timer */
    rc = timer_settime(cypress->mode_change_timerid, 0, &(cypress->itime), NULL);

    if (EOK != rc) {
        mtouch_error (cypress->log_name, "Failed to set mode chnage timer to %d.  Error: %d", nsec, rc);
        error_memory ("Cypress_Touch: Failed to set mode change timer to %d.  Error: %d", nsec, rc);
        return -1;
    }

    return EOK;
}

int cypress_controller_soft_reset(cypress_dev_t* cypress)
{
    uint8_t mode = UNKNOWN;
    int rc = 0;

    mode = cypress_get_mode(cypress);
    if (mode == BOOTLOADER_MODE) {
        mtouch_error(cypress->log_name, "Cypress controller already in bootloader mode, skip soft reset all OK !\n");
        return 0;
    }
    /* this part is added for to skip soft reset, if failed to read HST_MODE */
    else if (mode == 0xFF) {
        mtouch_error(cypress->log_name, "Cypress controller cypress_get_mode: HST_MODE i2c read failed, skip soft reset!\n");
        return -1;
    }
    else {
          //nothing
    }

    if (cypress->verbose > 4)
        mtouch_debug(cypress->log_name, "Performing soft reset\n");
    rc = cypress_soft_reset(cypress);
    if (EOK != rc) {
        cypress->driver_state = UNKNOWN;
        cypress_update_pps_status(cypress, "status::unknown");
        return -1;
    }
    return 0;
}

int cypress_controller_hard_reset(cypress_dev_t* cypress)
{
    int rc = 0;

    if (cypress->hard_reset) {
        if (cypress->platform.tp_reset) {
            /* Platform specific reset */
            rc = cypress->platform.tp_reset(cypress);
            if (EOK != rc) {
                cypress->driver_state = UNKNOWN;
                cypress_update_pps_status(cypress, "status::unknown");
                mtouch_error(cypress->log_name, "error performing platform-specific reset, errorno:%d, rc:%d", errno, rc);
                return -1;
            } else {
                return 0;
            }
        }
    }
    return -1;
}

int
cypress_controller_reset(cypress_dev_t* cypress)
{
    int rc;
    pthread_mutex_lock(&cypress->ctrl_reset_mutex);

    /* Notify PPS object that a reset has been triggered */
    cypress_update_pps_status(cypress, "status::reset");

    cypress->controller_reset_state = true;
    if (cypress->verbose > 4)
        mtouch_debug(cypress->log_name, "performing hard reset\n");
    rc = cypress_controller_hard_reset(cypress);
    if (rc != EOK)
    {
        mtouch_error(cypress->log_name, "hard reset failed, performing soft reset: %d \n", errno);
        rc = cypress_controller_soft_reset(cypress);
        if (rc != EOK)
        {
            mtouch_error(cypress->log_name, "soft reset failed %d\n", rc);
            pthread_mutex_unlock(&cypress->ctrl_reset_mutex);
            return -1;
        } else {
            if((usleep(cypress->reset_delay * 1000)) == 0)
                mtouch_debug(cypress->log_name, "executed soft reset successfully\n");
        }
    } else {
        if((usleep(cypress->reset_delay * 1000)) == 0)
            mtouch_debug(cypress->log_name, "executed hard reset successfully\n");
    }

    /* Platform specific clear irq if exists */
    if (cypress->platform.tp_clear_irq) {
        rc = cypress->platform.tp_clear_irq(cypress);
        if (EOK != rc) {
            mtouch_error(cypress->log_name, "Error performing platform-specific clear irq");
            error_memory("Cypress_Touch: Error performing platform-specific clear irq");
        }
    }

    /* Unmask interrupt */
    InterruptUnmask(cypress->tp_intr, cypress->tp_iid);

    /* Cleanup allocated memory */
    system_information_clean_up(cypress);

    /* Cypress suggested to use bootloader ACTIVE state as default bootloader state */
    cypress->bootloader_state = ACTIVE;
    cypress->driver_state = BOOTLOADER;

    cypress_update_pps_status(cypress, "status::bootloader");
    pthread_mutex_unlock(&cypress->ctrl_reset_mutex);

    return EOK;
}

int
cypress_driver_haptic_mode_check(cypress_dev_t* cypress)
{
    uint8_t reg[2] = {0};
    int rc = -1;
    int pendingCount = 0;


mode_change_retry:
    if ((rc = cypress->i2c_funcs.read_reg(cypress->i2c_fd, HST_MODE, 2, reg)) != 0) {
           mtouch_error (cypress->log_name, "Failed to read HST_MODE registers");
           return -1;
    } else {
        mtouch_info (cypress->log_name, "HST_MODE and RESET_DETECT Register read successful: mode -> %d %d",  reg[HST_MODE], reg[RESET_DETECT]);
    }

    if (reg[RESET_DETECT] == 0){

           //HST_MODE 3rd BIT checking//
           if((reg[HST_MODE] & MODE_CHANGE_SWITCH) != 0){
               mtouch_error (cypress->log_name, "cypress_driver_haptic_mode_check: Entered HST_MODE mode change bit not set");
               pendingCount++;
               delay(10);
               if(pendingCount < PENDING_COUNT_INFINEON) {
                  mtouch_error (cypress->log_name, "cypress_driver_haptic_mode_check: HST_MODE mode change bit not set, retrying: %d\n", pendingCount);
                  goto mode_change_retry;
               } else {
mode_re_request:
                  cypress->repeatCount++;
                  mtouch_error (cypress->log_name, "cypress_driver_haptic_mode_check: Enterd to re-request CAT mode, count: %d", cypress->repeatCount);
                  if(cypress->repeatCount < REPEAT_COUNT_INFINEON) {
 
                  /*reply back to set_haptic api for to re-request CAT mode*/
                  if ((cypress->haptic_flag) && (cypress->ext_recvid != -1)) {
                      rc = MsgReply(cypress->ext_recvid, CAT_MODE_CHANGE_RE_REQUEST, NULL, 0);
                      if (rc != EOK) {
                          mtouch_info (cypress->log_name, "failed to send re-trigger CAT mode request %d", rc);
                          MsgError(cypress->ext_recvid, -errno);
                      } else {
                          mtouch_info(cypress->log_name, "Cat mode re-trigger signal sent. rc %d\n", rc);
                      }
                      return EOK;
                  }

                 } else {
                     mtouch_error (cypress->log_name, "haptic CAT mode re-request max count reached, going to reset controller");
                     if (EOK != cypress_controller_reset(cypress)) {
                            mtouch_critical (cypress->log_name, "Failed to reset controller");
                     }
                     return -1;
                 }
               }
           } else {
               // Mode change bit is 0//
               mtouch_info(cypress->log_name, "HST_MODE mode change bit is zero, going to check the HST_MODE mode bit");
               if(reg[HST_MODE] == CONFIGURATION_AND_TEST_MODE) {
                  mtouch_info(cypress->log_name, "haptic: controller is successfully entered in to CAT mode");
                  if ((cypress->haptic_flag) && (cypress->ext_recvid != -1)) {
                      cypress->haptic_flag = false;
                      rc = MsgReply(cypress->ext_recvid, 0, NULL, 0);
                      if (rc != EOK) {
                         mtouch_info (cypress->log_name, "Failed to send Cat mode switch signal %d", rc);
                         MsgError(cypress->ext_recvid, -errno);
                      } else {
                         mtouch_info(cypress->log_name, "Cat mode switch signal sent. rc %d\n", rc);
                      }
                      cypress->ext_recvid = -1;
                  }
                  return EOK;
               } else{
                   mtouch_error(cypress->log_name, "haptic: mode change bit is zero, but controller not in CAT mode, going to re-request CAT mode");
                   goto mode_re_request;
               }
           }

    } else {
            mtouch_error (cypress->log_name, "cypress_driver_haptic_mode_check: Controller switched back to Bootloader mode: mode -> %d %d",  reg[HST_MODE], reg[RESET_DETECT]);
            error_memory("Cypress_Touch: Controller switched back to Bootloader mode: mode -> %d %d",  reg[HST_MODE], reg[RESET_DETECT]);

            rc = bootloader_state_machine(cypress);
            if (rc < 0)
                mtouch_error (cypress->log_name, "cypress_driver_haptic_mode_check: Bootloader exit command failed");
            else
                mtouch_info(cypress->log_name, "cypress_driver_haptic_mode_check: Bootloader exit command send successfully");

            return -1;
    }

    return rc;
}

int
cypress_driver_init(cypress_dev_t* cypress)
{
    uint8_t reg[3] = {0};
    uint8_t old_mode = 0;
    int skip_setting_mode = 0;
    int rc;

    switch (cypress->driver_state) {
    case BOOTLOADER:
        rc = bootloader_state_machine(cypress);
        if (rc < 0)
            goto fail2;
        break;
    case SYSTEM_INFORMATION:
        /* Stop bootloader watchdog */
        cypress_set_boot_watchdog_timer(cypress, 0);
        /* Check to see if we are out of boot loader mode */
        /* During testing it was found that sometimes after moving to SYS INFO mode, value is not stable, (receive 255 1), adding 200ms delay to stabilize */
        (void)usleep(200000);    // Casting to void to supress coverity warning
        if ((rc = cypress->i2c_funcs.read_reg(cypress->i2c_fd, HST_MODE, 2, reg)) != 0) {
            mtouch_error (cypress->log_name, "Failed to read bootloader registers");
            goto fail2;
        }

        if (((reg[HST_MODE] & 0x01) != 0) && (reg[RESET_DETECT] != 0)) {
            /* Still in bootloader, may need to execute a bootload */
               mtouch_critical (cypress->log_name, "Unable to exit bootloader: mode -> %d %d",  reg[HST_MODE], reg[RESET_DETECT]);
               error_memory("Cypress_Touch: Unable to exit bootloader: mode -> %d %d",  reg[HST_MODE], reg[RESET_DETECT]);
               mtouch_critical (cypress->log_name, "No valid application available on controller, update controller firmware.");
               error_memory("Cypress_Touch: No valid application available on controller, update controller firmware.");
               cypress->driver_state = BOOTLOADER;
               cypress->bootloader_state = ACTIVE;

               cypress_update_pps_status(cypress, "status::bootloader");
               return -1;
        }

        if ((reg[HST_MODE] & 0x10) != 0x10)
            return 0;

        /* We are in System Information mode */
        if (cypress->verbose > 4)
            mtouch_info (cypress->log_name, "Controller is in System Information mode, reading memory map");

        /* Read All System Information */
        if (cypress_read_memory_map (cypress) != EOK)
            goto fail2;

        /* Put controller into Operating Mode */
        if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, HST_MODE, 2, reg)) == 0) {
            if (reg[RESET_DETECT] == 0) {
                rc = cypress_set_mode(cypress, OPERATING_MODE);
                if (rc != -1)
                {
                    cypress->controller_reset_state = false;
#ifdef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
                    cypress->sysinfo_to_operating_mode_state = true;
#endif
                }
            } else {
                mtouch_error (cypress->log_name, "Register 1 is not 0!!!! Not sure what to do with this yet.");
                goto fail2;
            }
        } else {
            mtouch_error (cypress->log_name, "Failed to read bootloader registers");
            goto fail1;
        }
        mtouch_info(cypress->log_name, "Complete init succeed, retry attempts to zero");
        cypress->retry_init_attempts = 0;
        cypress->driver_state = OPERATING;
        cypress_update_pps_status(cypress, "status::operating");

        /* Read flash params for configuration details */
        //cypress_read_config_param_flash(cypress, 0, NULL, 0);

        /* We are are running both tests set the mode now and save ourselves some time ~100ms */
        if ((cypress->run_cm_panel_test) && (cypress->run_cp_panel_test)) {
            /* Ensure we are in Config Test Mode */
            if ((old_mode = cypress_get_mode(cypress)) != CONFIGURATION_AND_TEST_MODE) {
                rc = cypress_set_mode(cypress, CONFIGURATION_AND_TEST_MODE);
                if (-1 == rc) {
                    mtouch_error (cypress->log_name, "Failed to set controller to config and test mode");
                    error_memory ("Cypress_Touch: Failed to set controller to config and test mode");
                    return -1;
                }
            }
            skip_setting_mode = 1;
        }

        /* Check to see if the driver needs to perform any Manufacturering tests */
        if ((cypress->run_cp_panel_test) || (cypress->save_cm_cp_data))
            cypress_run_cp_panel_test(cypress, skip_setting_mode, cypress->save_cm_cp_data);

        if ((cypress->run_cm_panel_test) || (cypress->save_cm_cp_data))
            cypress_run_cm_panel_test(cypress, skip_setting_mode, cypress->save_cm_cp_data);

        /* Restore Old Mode if applicable */
        if ((cypress->run_cm_panel_test) && (cypress->run_cp_panel_test)) {
            if (old_mode != CONFIGURATION_AND_TEST_MODE) {
                rc = cypress_set_mode(cypress, (old_mode));
                if (-1 == rc) {
                    mtouch_error (cypress->log_name, "Failed to restore controller mode after config test: %s", __FUNCTION__);
                    error_memory("Cypress_Touch: Failed to restore controller mode after config test: %s", __FUNCTION__);
                    return -1;
                }
            }
        }

        /* Check to see if there is a pending IDAC calibration */
        if (cypress->calibration_required) {
            cypress_calibrate_idacs(cypress);
            cypress->calibration_required = 0;
        } else if (cypress->sensing_mode)
            /* Always Initialize Baselines */
            cypress_init_baselines(cypress);

        /* System should be all set to go, start the heartbeat timer */
        if (EOK != cypress_set_heartbeat_timer(cypress, cypress->heartbeat_sleep)) {
            mtouch_error (cypress->log_name, "%s failed to set heartbeat timer", __FUNCTION__);
            goto fail2;
        }
        cypress->cy_pm_state = PM_STATE_RESUME;
        mtouch_info (cypress->log_name, "cypress device initialized successfully");

        break;
    case CONFIGURATION_AND_TEST_MODE:

            mtouch_info(cypress->log_name, "Received Interrupt and entered in CAT mode case");

            /* Stop mode change timer */
            mtouch_info (cypress->log_name, "mode change timer stopped");
            if (EOK != cypress_set_mode_change_timer(cypress, 0)) {
                mtouch_error (cypress->log_name, "%s failed to set mode change timer", __FUNCTION__);
            }

            rc = cypress_driver_haptic_mode_check(cypress);

            if((rc != EOK) && cypress->haptic_flag){
               mtouch_error(cypress->log_name, "%s haptic CAT mode request failed, replying caller error: %d", __FUNCTION__, rc);
               if ((cypress->haptic_flag) && (cypress->ext_recvid != -1)) {
               cypress->haptic_flag = false;
               rc = MsgReply(cypress->ext_recvid, -1, NULL, 0);
               if (rc != EOK) {
                  mtouch_info (cypress->log_name, "Failed to send Cat mode switch signal %d", rc);
                  MsgError(cypress->ext_recvid, -errno);
               } else {
                  mtouch_info(cypress->log_name, "Cat mode switch signal sent. rc %d\n", rc);
               }
               cypress->ext_recvid = -1;
               }
            }

         break;
    default:
        break;
    }

    return EOK;

fail1:
    system_information_clean_up(cypress);
fail2:
    return -1;
}

/**
 * Attach to libinputevents
 */
static int
attach_driver(cypress_dev_t* cypress)
{
    mtouch_driver_funcs_t funcs = {
        .get_contact_id = get_contact_id,
        .is_contact_down = is_contact_down,
        .get_coords = get_coords,
        .get_down_count = NULL,
        .get_touch_width = NULL,
        .get_touch_height = NULL,
        .get_touch_orientation = NULL,
        .get_touch_pressure = get_touch_pressure,
        .get_seq_id = get_seq_id,
        .set_event_rate = NULL,
        .get_contact_type = NULL,
        .get_select = NULL
    };

    mtouch_driver_params_t params = {
        .capabilities = MTOUCH_CAPABILITIES_CONTACT_ID |
                        MTOUCH_CAPABILITIES_COORDS |
                        MTOUCH_CAPABILITIES_PRESSURE,
        .flags = 0,
        .max_touchpoints = cypress->num_touchpts,
        .width = cypress->width,
        .height = cypress->height
    };

    cypress->inputevents_hdl = mtouch_driver_attach(&params, &funcs);

    if (NULL == cypress->inputevents_hdl) {
        mtouch_error(cypress->log_name, "Failed to connect to libinputevents");
        error_memory("Cypress_Touch: Failed to connect to libinputevents");
        return -1;
    }

    return EOK;
}



/**
 * Loop the I2C controller and TP reset
 * @param error Non-zero if the reset was due to an error
 */
int
reset_all_loop(cypress_dev_t* cypress, int error)
{
    int i;
    int rc;

    for (i = 0; i < CYPRESS_MAX_RESET_ATTEMPTS; i++) {
        rc = reset_all(cypress, error);
        if (EOK == rc) {
            /* We recovered */
            return EOK;
        }
    }

    /* Unable to reset to a recoverable state */
    mtouch_critical(cypress->log_name, "Failed to reset I2C and TP after %d attempts, giving up", CYPRESS_MAX_RESET_ATTEMPTS);
    error_memory("Cypress_Touch: Failed to reset I2C and TP after %d attempts, giving up", CYPRESS_MAX_RESET_ATTEMPTS);
    return rc;
}

uint16_t
cypress_extract_data(cypress_dev_t* cypress, uint8_t *buf, uint8_t byte_offset, uint8_t bit_offset, uint8_t num_bits)
{
    uint8_t i;
    uint16_t mask = 0;
    uint16_t val = 0;

    for (i = 0; i < num_bits; i++)
        mask |= (1 << i);

    if (num_bits > 8) {
        /* Get the High byte */
        val = (buf[byte_offset] & mask) << 8;
        val |= buf[byte_offset + 1];
    } else {
        val |= ((buf[byte_offset] >> bit_offset) & mask);
    }

    return val;
}

void cypress_dump_rtd_record(cypress_dev_t* cypress, uint8_t *data)
{
    char *buf;
    int i;

    buf = (char *) malloc (MAX_RTD_DATA_LENGTH);
    if (buf == NULL) {
        mtouch_error (cypress->log_name, "Failed to allocate buffer space to dump rtd record");
        return;
    }

    for (i = 0; i < MAX_RTD_DATA_LENGTH; i++) {
        sprintf ((buf + i), "%x ", data[i]);
    }

   mtouch_debug(cypress->log_name,"rtd fail status = 0x%x", data[RTD_FAIL_STATUS]);
   mtouch_debug(cypress->log_name,"rtd counter = 0x%x", data[RTD_COUNTER]);
   if (cypress->verbose > 7)
        mtouch_debug(cypress->log_name,"rtd data = %s", buf);

    free(buf);
}


int cypress_dump_touch_record (cypress_dev_t* cypress, uint8_t *touch_record, int size)
{
    char *buf;
    int i;

    buf = (char *) malloc (size);

    if (buf == NULL) {
        mtouch_error (cypress->log_name, "Failed to allocate buffer space to dump touch record");
        return -1;
    }

    sprintf (buf, "%x ", touch_record[0]);

    for (i = 1; i < size; i++) {
        sprintf ((buf + strlen(buf)), "%x ", touch_record[i]);
    }

    mtouch_debug (cypress->log_name, "Touch Record [%d] Status [%d]: %s",
    (uint8_t)cypress_extract_data (cypress, touch_record, (cypress->register_map.operating_mode_info->tch_rec_6 & 0x1f),
                                ((cypress->register_map.operating_mode_info->tch_rec_6 & 0xE0) >> 5),
                                cypress->register_map.operating_mode_info->tch_rec_7),
    (uint8_t)cypress_extract_data(cypress, touch_record, (cypress->register_map.operating_mode_info->tch_rec_8 & 0x1f),
                                ((cypress->register_map.operating_mode_info->tch_rec_8 & 0xE0) >> 5),
                                cypress->register_map.operating_mode_info->tch_rec_9),
    buf);

    free (buf);

    return 0;
}

int cypress_process_button(cypress_dev_t* cypress)
{
    int i, j;
    uint16_t idx = -1;
    uint8_t *button_record = 0;
    int rc;

    if (!cypress->register_map.operating_mode_info->num_btns) {
        if (cypress->verbose > 8)
            mtouch_info(cypress->log_name, "buttons are not configured");
        return EOK;
    }

    button_record = (uint8_t *)calloc (1, sizeof(uint8_t));
    idx = 0;
    for (i = 0; i < cypress->num_button_regs; i++) {
        if ((rc = cypress->i2c_funcs.read_reg(cypress->i2c_fd, (cypress->register_map.operating_mode_info->rep_ofs + (BUTTONS0 + i)), 1, button_record)) < 0 ) {
                mtouch_error(cypress->log_name, "failed to read button record");
                return rc;
        }

        if (cypress->verbose > 7)
            mtouch_info (cypress->log_name, "RAW Button information for group %d: %x", i, *button_record);

        for (j = 0; j < 4; j++) {
            cypress->capsense_button_state[idx] = (((*button_record) >> (2 * j)) & 0x03);
            idx++;
        }
    }

    if (cypress->verbose > 6) {
        for (i = 0; i < cypress->register_map.operating_mode_info->num_btns; i++)
            mtouch_info (cypress->log_name, "Button %d status: %s", i, cypress->capsense_button_state[i] ? "Pressed" : "Released");
    }

    idx = -1;
    if (memcmp (cypress->capsense_button_state, cypress->capsense_button_old_state, sizeof (cypress->capsense_button_state))) {
        /* Notify external app of any state change. */
        if (-1 == MsgDeliverEvent(cypress->app_rcvid, &(cypress->app_notify_event))) {
            mtouch_error (cypress->log_name, "MsgDeliverEvent failed (%s) Unable to alert external app of button status change",strerror(errno));
            error_memory ("Cypress_Touch: MsgDeliverEvent failed (%s) Unable to alert external app of button status change",strerror(errno));
        }
    }
    memcpy (cypress->capsense_button_old_state, cypress->capsense_button_state, sizeof (cypress->capsense_button_state));

    free(button_record);
    return EOK;
}

int cypress_get_display_grpid(cypress_dev_t* cypress)
{
    int handle, ret;
    char *reply;

    reply = NULL;

    if(!cypress->dev_status_path)
    {
        mtouch_error("Not a valid status path %s", cypress->dev_status_path);
        return -EINVAL;
    }
    handle = displaybinder_status_open(cypress->dev_status_path);
    if(handle < 0)
    {
        mtouch_error(cypress->log_name, "Failed to open display binder -Dev status path: %s, handle:%d",cypress->dev_status_path, handle);
        return handle;
    }
    mtouch_debug(cypress->log_name,"display binder open success(reval: %d) for dev status path %s", handle, cypress->dev_status_path);
     ret = displaybinder_status_get(handle,"display-group-id",&reply);
    displaybinder_status_close(handle);
    if(ret < 0)
    {
       mtouch_error(cypress->log_name,"failed to fetch group id: %d errno:%d", ret, errno);
    } else {
       cypress->display_group_id = atoi(reply);
       mtouch_info (cypress->log_name, "Display Group ID is fetched successfully: %d", cypress->display_group_id);
    }

   if(reply)
      free(reply);
   return ret;
}
int cypress_process_rtd(cypress_dev_t* cypress)
{
    int ret;
    operating_mode_info_t *op_info = cypress->register_map.operating_mode_info;
    uint8_t *buf;

    buf =(uint8_t *) calloc(op_info->rtd_rec_siz + 1, sizeof(uint8_t));
    if (buf == NULL) {
        mtouch_error(cypress->log_name,"failed to allocate memory for rtd record, rc= -%d", ENOMEM);
        return -ENOMEM;
    }

    ret = cypress->i2c_funcs.read_reg(cypress->i2c_fd, op_info->rtd_rec_ofs, op_info->rtd_rec_siz, buf);
    if (ret != 0) {
        mtouch_error(cypress->log_name,"failed to read rtd record, rc = %d", ret);
        return ret;
    }

    if(cypress->display_group_id < 0)
    {
        ret = cypress_get_display_grpid(cypress);
        if(ret != 0)
        {
           mtouch_error(cypress->log_name,"failed to fetch group id: %d again", cypress->display_group_id);
           return -1;
        }
    }
    buf[MAX_RTD_DATA_LENGTH - 1] = cypress->display_group_id;
    memcpy(cypress->rtd_data, buf, MAX_RTD_DATA_LENGTH);

    cypress->rtd_len = MAX_RTD_DATA_LENGTH;
    if (cypress->verbose > 6)
        cypress_dump_rtd_record(cypress, buf);

    free(buf);
    return EOK;
}

void cypress_release_touch(cypress_dev_t* cypress)
{
    int Pending_release = 0;
    int count = 0;

    if (cypress == NULL)
        return;

      /* Check and compare pending release events mapped to index */
    for(count = 0; count < CYPRESS_NUM_TOUCHPOINTS; count++)
    {
        if(cypress->num_prev_recs[count] == 1)
        {
            Pending_release++;
            cypress->touch_report[count].status = 0;
            cypress->num_prev_recs[count] = 0;
            mtouch_error(cypress->log_name, "Release ID %d Coordinate X: %d Y: %d P: %d status: %s", count+1,
                                                                            cypress->touch_report[count].x,
                                                                            cypress->touch_report[count].y,
                                                                            cypress->touch_report[count].pressure,
                                                                            (cypress->touch_report[count].status ? "Pressed" : "Released"));
        }
    }
     /* If there is any pending release then process the packet */
    if(Pending_release > 0)
    {
        mtouch_error(cypress->log_name, "cypress_release_touch entered - Pending Release - %d\n",Pending_release);
        mtouch_driver_process_packet(cypress->inputevents_hdl, &cypress->touch_report[0], cypress, MTOUCH_PARSER_FLAG_NONE);
        mtouch_error(cypress->log_name, "Released all %d touch pointers by process packet", Pending_release);
    }
}

int cypress_process_touch(cypress_dev_t* cypress, int num_touch_records)
{
    uint8_t *touch_record = 0;
    int i;
    int16_t idx = -1;
    operating_mode_info_t *op_info = cypress->register_map.operating_mode_info;
    int rc;

    for (i = 0; i < num_touch_records; i++) {
        touch_record =(uint8_t *) calloc(cypress->register_map.operating_mode_info->tch_rec_siz, sizeof(uint8_t));

        if ((rc = cypress->i2c_funcs.read_reg(cypress->i2c_fd,
                                         (op_info->tt_stat_ofs + 1 + (op_info->tch_rec_siz * i)),
                                         op_info->tch_rec_siz,
                                         touch_record)) != 0) {
            mtouch_error(cypress->log_name, "failed to touch record info from the controller ");
            free(touch_record);  // Free memory before exiting
            goto fail;
        }

        idx = (uint8_t)cypress_extract_data(cypress, touch_record,
                                            (op_info->tch_rec_6 & 0x1f),
                                            ((op_info->tch_rec_6 & 0xE0) >> 5),
                                             op_info->tch_rec_7);

        if(idx >= CYPRESS_NUM_TOUCHPOINTS)
        {
            rc = 1;
            mtouch_error(cypress->log_name, "Invalid touch index %d", idx);
            free(touch_record);  // Free memory before exiting
            goto fail;
        }
        cypress->touch_report[idx].status = (uint8_t)cypress_extract_data(cypress, touch_record,
                                                                          (op_info->tch_rec_8 & 0x1f),
                                                                          ((op_info->tch_rec_8 & 0xE0) >> 5),
                                                                          op_info->tch_rec_9);
        if((cypress->touch_report[idx].status) == TOUCH_PRESS)
        {
            cypress->num_prev_recs[idx] = 1;
        }

        switch (cypress->touch_report[idx].status) {
        case NO_EVENT:
            /* No Event means that the finger is still down, there is more than
            one finger pressed and this finger hasn't moved */
        case TOUCH_PRESS:
        case TOUCH_MOVE:
            /* Move */
            cypress->touch_report[idx].status = 1;  /* Note: This is intentionally the same as Press,
                                                as libinput events doesn't have a way to use this data yet */
            break;
        case TOUCH_RELEASE:
            /* Release */
            cypress->touch_report[idx].status = 0;
            cypress->num_prev_recs[idx] = 0;
            break;
        default:
            break;
        }

        cypress->touch_report[idx].x = cypress_extract_data(cypress, touch_record, op_info->tch_rec_0, 0, op_info->tch_rec_1);
        cypress->touch_report[idx].y = cypress_extract_data(cypress, touch_record, op_info->tch_rec_2, 0, op_info->tch_rec_3);
        cypress->touch_report[idx].pressure = cypress_extract_data(cypress, touch_record, op_info->tch_rec_4, 0, op_info->tch_rec_5);
        cypress->touch_report[idx].width = cypress_extract_data(cypress, touch_record, (op_info->tch_rec_14 & 0x1f),
                                                                ((op_info->tch_rec_14 & 0xE0) >> 5 ),op_info->tch_rec_15);
        cypress->touch_report[idx].height = cypress_extract_data(cypress, touch_record, (op_info->tch_rec_16 & 0x1f),
                                                                ((op_info->tch_rec_16 & 0xE0) >> 5 ), op_info->tch_rec_17);
        cypress->touch_report[idx].orientation = cypress_extract_data(cypress, touch_record, (op_info->tch_rec_18 & 0x1f),
                                                                ((op_info->tch_rec_18 & 0xE0) >> 5 ), op_info->tch_rec_19);

        mtouch_info (cypress->log_name, "RAW Finger %d Coordinate X: %d Y: %d P: %d status: %s", idx,
                                                                                cypress->touch_report[idx].x,
                                                                                cypress->touch_report[idx].y,
                                                                                cypress->touch_report[idx].pressure,
                                                                                (cypress->touch_report[idx].status ? "Pressed" : "Release"));

        if (cypress->verbose > 4)
        {
            cypress_dump_touch_record (cypress, touch_record, op_info->tch_rec_siz);
        }
        free(touch_record);  // Free memory at the end of each iteration
    }

    if (-1 != idx) {
        /* Process the data now */
        mtouch_driver_process_packet(cypress->inputevents_hdl, &cypress->touch_report[0], cypress, MTOUCH_PARSER_FLAG_NONE);

        if (cypress->release_timeout) {
            memcpy (cypress->last_touch_report, cypress->touch_report, (cypress->num_touchpts * sizeof(touch_report_t)));
        }
    }
    rc = EOK;
fail:
    return rc;
}

int
cypress_read_touch_data(cypress_dev_t* cypress)
{
    uint8_t rep_stat_reg = 0;
    uint8_t ttstat_reg = 0;
    uint8_t rtd_update = 0;
    uint8_t rep_len = 0;
    int num_touch_records = 0;
    int noise_interferance = 0;
    int ret;

    /* Check the current hardware mode and ensure we are in Operating Mode
       so the driver and hardware are in sync */
    switch (cypress_get_mode (cypress)) {
    case OPERATING_MODE:
        break;
    default:
        /* ESD might have put us back into bootloader mode or something unexpected. Reset and start again. */
        mtouch_error(cypress->log_name, "While reading touch data ,controller is in the wrong state, resetting...");
        error_memory("Cypress_Touch: Controller is in the wrong state, resetting...");

        if (EOK != cypress_controller_reset(cypress)) {
            mtouch_error(cypress->log_name, "Failed to reset the touch controller");
        }

        goto fail;
    }

    if (cypress->register_map.operating_mode_info != NULL) {

        /* read report len register */
        if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, (cypress->register_map.operating_mode_info->rep_ofs), 1, &rep_len)) != 0) {
            mtouch_error (cypress->log_name, "Error reading report len register");
            goto fail;
        }

        /* Check Report Status Register */
        if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, (cypress->register_map.operating_mode_info->rep_ofs + REP_STAT), 1, &rep_stat_reg)) != 0) {
            mtouch_error (cypress->log_name, "Error reading report status register");
            goto fail;
        }

        /* read tt_stat register */
        if ((cypress->i2c_funcs.read_reg(cypress->i2c_fd, cypress->register_map.operating_mode_info->tt_stat_ofs, 1, &ttstat_reg)) != 0) {
            mtouch_warn(cypress->log_name, "Unable to read the number of pending reports on controller");
            goto fail;
        }
    }
    else {
        mtouch_error (cypress->log_name, "operating_mode_info register is NULL; unable to read touch_data");
        goto fail;
    }

    /* Check Noise Effects and report any effects */
    if (cypress->verbose > 3) {
        switch (rep_stat_reg & REP_NOISE_EFTS) {
        case 0:
            if (cypress->verbose > 7) {
                mtouch_info (cypress->log_name, "Noise level is insignificant and having no effect on system performance.");
            }
            break;
        case 1:
            mtouch_info (cypress->log_name, "Increased processing. No reduction in refresh rate.");
            break;
        case 2:
            mtouch_info (cypress->log_name, "Refresh rate reduced to 85 percent or more of configured refresh rate");
            break;
        case 3:
            mtouch_info (cypress->log_name, "Refresh rate reduced to less than 85 percent of configured refresh rate.");
            break;
        case 4:
            mtouch_info (cypress->log_name, "Change FH configuration cycles.");
            break;
        // 5 & 6 Reserved
        case 7:
            mtouch_info (cypress->log_name, "The noise level is significant enough that reliable touch data cannot be obtained.");
            noise_interferance = 1;
            break;
        default:
            break;
        }
    }

    rtd_update = rep_stat_reg & REP_RTD_UPDATE;
    num_touch_records = ttstat_reg & 0x1F;

    if (noise_interferance) {
        /* Don't use the touch information if there was any noise interferance */
        num_touch_records = 0;
    }

    /* Check to see if the report is valid */
    if (rep_stat_reg & REP_INVALID) {
        if (cypress->verbose > 1)
            mtouch_info (cypress->log_name, "Received an invalid report, ignoring.");
        return EOK;
    }

    /* check to see valid report length */
    if ((rep_len == 0) && (num_touch_records > 0)){
        mtouch_error(cypress->log_name," report length error rep_len=%d num_tch=%d",
                     rep_len, num_touch_records);
        return EOK;
    }

    /* check to see large area detected */
    if (ttstat_reg & 0x20) {
        /* Do not report touch if configured so */
        mtouch_info (cypress->log_name, "Large area detected, ignoring touch data");
        num_touch_records = 0;
    }

    if (num_touch_records > cypress->register_map.operating_mode_info->max_tchs) {
        mtouch_error(cypress->log_name," %s (n=%d c=%zu)",
                     "too many tch; set to max tch",
                     num_touch_records, cypress->register_map.operating_mode_info->max_tchs);
        num_touch_records = cypress->register_map.operating_mode_info->max_tchs;
    }

    /* Get Button information (if present) */
    ret = cypress_process_button(cypress);
    if (ret)
        return ret;

    /* Get touch information */
    if ((0 == num_touch_records) && (rtd_update == 0)) {
        if (cypress->verbose > 5)
            mtouch_debug(cypress->log_name, "No touch record, no RTD UPDATE: %x %x", ttstat_reg, rep_stat_reg);
        /* no touch records and no rtd record -> ignore the message */
        return EOK;
    }

    if ((rtd_update) && (cypress->rtd_enable == 1)){

        /* touch records or rtd record */
        /* print RTD_UPDATE bit to check touch record or rtd record */
        if (cypress->verbose > 5) {
            mtouch_debug(cypress->log_name, "Raw read of REP_STAT, RTD UPDATE: %x %x", rep_stat_reg, rtd_update);
            mtouch_info(cypress->log_name, "Raw read of Operating mode info TT_STAT: %x", ttstat_reg);
        }

        /* process rtd record */
        ret = cypress_process_rtd(cypress);
        if (ret == EOK) {

            /* Stop the heartbeat timer */
            if (EOK != cypress_set_heartbeat_timer(cypress, 0)) {
                mtouch_error (cypress->log_name, "%s failed to set heartbeat timer", __FUNCTION__);
            }

            cypress->rtd_readflag = 1;
            cypress->controller_reset_count = 0;

            /* (Re)Start heartbeat timer */
            if (EOK != cypress_set_heartbeat_timer(cypress, cypress->heartbeat_sleep)) {
                mtouch_error (cypress->log_name, "%s failed to set heartbeat timer", __FUNCTION__);
            }
        }
        else {
            mtouch_error(cypress->log_name, "failed to processs rtd, ret: %d", ret);
        }

    }

    if (num_touch_records) {
        /* process num_touch_records */
        ret = cypress_process_touch(cypress, num_touch_records);
        if (ret)
            return ret;
    } else {
        /* release any previous press touch events */
        cypress_release_touch(cypress);
    }

    return EOK;
fail:
    return -1;
}

int
cypress_existence_check(cypress_dev_t* cypress)
{
    uint8_t* byte = 0;
    int error;

    error = cypress->i2c_funcs.read_reg(cypress->i2c_fd, HST_MODE, 1, byte);

    if(error < 0) {
        mtouch_info(cypress->log_name, "Device existence check failed, %d, %d", error, errno);
        goto fail;
    }

    mtouch_info (cypress->log_name, "Cypress device found.");

    return EOK;

fail:
    return -1;
}

#ifdef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
bool cypress_process_mode_change_interrupt(cypress_dev_t* cypress)
{
    bool ret = false;
    int rc = -1;

    if (cypress->sysinfo_to_operating_mode_state)
    {
        ret = true;
        if (cypress_get_mode(cypress) == OPERATING_MODE)
        {
            mtouch_info(cypress->log_name, "Controller mode change from Sysinfo to Operating success");
            cypress->sysinfo_to_operating_mode_state = false;
        }
        else
        {
            mtouch_info(cypress->log_name, "Controller mode change from Sysinfo to Operating failed");
        }
    }
    else if (cypress->cat_to_operating_mode_state)
    {
        ret = true;
        if (cypress_get_mode(cypress) == OPERATING_MODE)
        {
            mtouch_info(cypress->log_name, "Controller mode change from CAT to Operating success");
            cypress->cat_to_operating_mode_state = false;
            /* Stop mode change timer */
            mtouch_info(cypress->log_name, "Mode change timer stopped");
            if (EOK != cypress_set_mode_change_timer(cypress, 0))
            {
                mtouch_error(cypress->log_name, "%s Failed to set mode change timer", __FUNCTION__);
            }
            rc = MsgReply(cypress->ext_recvid, 0, NULL, 0);
            if (rc != EOK)
            {
                mtouch_info(cypress->log_name, "Failed to send Operating mode switch signal rc:%d, errno:%d", rc, errno);
                MsgError(cypress->ext_recvid, -errno);
            } else
            {
                mtouch_info(cypress->log_name, "Operating mode switch signal sent. rc:%d\n", rc);
            }
        }
        else
        {
            rc = MsgReply(cypress->ext_recvid, -1, NULL, 0);
            if (rc != EOK)
            {
                mtouch_error(cypress->log_name, "Failed to send Operating mode switch failure rc:%d errno: %d", rc, errno);
                MsgError(cypress->ext_recvid, -errno);
            } else{
                mtouch_info(cypress->log_name, "Controller mode change from CAT to Operating failure reply sent succesfully rc:%d\n", rc);
            }
        }
    }
    return ret;
}
#endif

void*
tp_recv_thread(void* arg)
{
    cypress_dev_t* cypress =(cypress_dev_t *) arg;
    int rcvid;
    cypress_msg_u msg;
    int rc;
    struct sched_param thread_param;
    int i;
    int old_mode = 0;
    struct _msg_info  info;

    /* Do any platform specific initialization */
    if (EOK != cypress_platform_init(cypress)) {
        return NULL;
    }

    if (EOK != (pthread_getschedparam(pthread_self(), NULL, &thread_param))) {
        mtouch_error(cypress->log_name, "Failed to fetch the thread priority: %d", errno);
        return NULL;
    }

    if (cypress->verbose) {
        mtouch_info (cypress->log_name, "Starting touch receive thread at prio %d", thread_param.sched_curpriority);
    }

    /* Create the heartbeat/ping timer */
    cypress->heartbeat_event.sigev_notify = SIGEV_PULSE;
    cypress->heartbeat_event.sigev_coid = ConnectAttach(0, 0, cypress->attach->chid, _NTO_SIDE_CHANNEL, 0);
    cypress->heartbeat_event.sigev_priority = thread_param.sched_curpriority;
    cypress->heartbeat_event.sigev_code = CYPRESS_HEARTBEAT_PULSE_CODE;
    timer_create(CLOCK_REALTIME, &(cypress->heartbeat_event), &(cypress->heartbeat_timerid));

    /* Create the bootloader watchdog timer */
    cypress->boot_watchdog_event.sigev_notify = SIGEV_PULSE;
    cypress->boot_watchdog_event.sigev_coid = ConnectAttach(0, 0, cypress->attach->chid, _NTO_SIDE_CHANNEL, 0);
    cypress->boot_watchdog_event.sigev_priority = thread_param.sched_curpriority;
    cypress->boot_watchdog_event.sigev_code = CYPRESS_BOOT_WATCHDOG_PULSE_CODE;
    timer_create(CLOCK_REALTIME, &(cypress->boot_watchdog_event), &(cypress->boot_watchdog_timerid));

    /* Create the mode change monitor timer */
    cypress->mode_change_event.sigev_notify = SIGEV_PULSE;
    cypress->mode_change_event.sigev_coid = ConnectAttach(0, 0, cypress->attach->chid, _NTO_SIDE_CHANNEL, 0);
    cypress->mode_change_event.sigev_priority = thread_param.sched_curpriority;
    cypress->mode_change_event.sigev_code = CYPRESS_MODE_CHG_TIMEOUT_PULSE_CODE;
    timer_create(CLOCK_REALTIME, &(cypress->mode_change_event), &(cypress->mode_change_timerid));

    /* Create release timer */
    if ((cypress->release_timeout) && (cypress->num_touchpts <= 4)) {
        cypress->inject_release_timerid =(timer_t *) malloc (cypress->num_touchpts * sizeof(timer_t));

        if (NULL == cypress->inject_release_timerid) {
            mtouch_error(cypress->log_name, "Failed to allocate memory for release timers");
            return NULL;
        }

        cypress->last_touch_report =(touch_report_t *) malloc (cypress->num_touchpts * sizeof(touch_report_t));

        if (NULL == cypress->last_touch_report) {
            mtouch_error(cypress->log_name, "Failed to allocate memory for last touch report");
            return NULL;
        }

        cypress->inject_release_event.sigev_notify = SIGEV_PULSE;
        cypress->inject_release_event.sigev_coid = ConnectAttach(0, 0, cypress->attach->chid, _NTO_SIDE_CHANNEL, 0);
        cypress->inject_release_event.sigev_priority = thread_param.sched_curpriority;

        for (i = 0; i < cypress->num_touchpts; i++) {
            cypress->inject_release_event.sigev_code = (CYPRESS_RELEASE_TIMEOUT_EVENT_PULSE_CODE + i);
            timer_create(CLOCK_REALTIME, &(cypress->inject_release_event), &(cypress->inject_release_timerid[i]));
        }
    } else if ((cypress->release_timeout) && (cypress->num_touchpts > 4)) {
        mtouch_error(cypress->log_name, "Release timer only supports upto 4 fingers, controller configured for %d fingers.  Timer not enabled.", cypress->num_touchpts);
        error_memory("Cypress_Touch: Release timer only supports upto 4 fingers, controller configured for %d fingers.  Timer not enabled.", cypress->num_touchpts);
        cypress->release_timeout = 0;
    }

    while (1) {
        rcvid = MsgReceive(cypress->attach->chid, &msg, sizeof(msg), &info);
        if (rcvid < 0) {
            if ((EINTR == errno) || (ETIMEDOUT == errno)) {
                continue;
            }

            //mtouch_error(cypress->log_name, "MsgReceive failed: %s", strerror(errno));
            error_memory("Cypress_Touch: MsgReceive failed: %s", strerror(errno));
            return NULL;
        }

        /* if cypress is in suspend/prepare state, Do not process any request except unmaksing TP_PULSE */
        if ((cypress->cy_str_state == PM_STATE_SUSPEND)) {
                if((msg.pulse.code == CYPRESS_TP_PULSE_CODE) || (msg.pulse.code == CYPRESS_TP_ERROR_PULSE_CODE))
                {
                    mtouch_info(cypress->log_name, "Received Interrupt in Suspend pm[%d]code[%d]",cypress->cy_pm_state, msg.pulse.code);
                    InterruptUnmask(cypress->tp_intr, cypress->tp_iid);
                    continue;
                }
        }else if (rcvid == 0) {
            /* Pulse received */
            switch (msg.pulse.code) {
            case _PULSE_CODE_DISCONNECT:
                ConnectDetach (msg.pulse.scoid);
                break;
            case CYPRESS_TP_PULSE_CODE:
                if (cypress->verbose > 6) {
                    mtouch_info (cypress->log_name, "Interrupt");
                }
                rc = pthread_mutex_lock(&cypress->tp_event_mutex);
                if (rc) {
                    mtouch_error(cypress->log_name, "Can't lock tp_event_mutex. Err: %d\n", rc);
                }

                if (cypress->driver_state != OPERATING) {
#ifdef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
                    if(cypress->haptic_write_status == true)
                    {
                        cypress->haptic_write_status = false;
                        mtouch_info(cypress->log_name, "Ignore the interrupt, as it's for haptic write");
                    }
                    else
                    {
#endif
                        if (UNKNOWN == cypress->driver_state) {
                            mtouch_error (cypress->log_name, "CYPRESS_TP_PULSE_CODE controller in unknown mode, resetting ...");
                            if (EOK != cypress_controller_reset(cypress)) {
                                mtouch_critical (cypress->log_name, "Failed to reset controller");
                            }
                        }

                        if (-1 == cypress_driver_init(cypress)) {
                            mtouch_critical (cypress->log_name,"Driver init failed, retrying %d", cypress->retry_init_attempts);
                            error_memory("Cypress_Touch: Driver init failed, retrying %d", cypress->retry_init_attempts);
                            cypress->retry_init_attempts++;
                            delay (1000);
                            mtouch_error (cypress->log_name, "CYPRESS_TP_PULSE_CODE init call failed, resetting ...");

                            if (EOK != cypress_controller_reset(cypress)) {
                                mtouch_critical (cypress->log_name, "Failed to reset controller");
                            }
                        } else {
                            mtouch_info(cypress->log_name, "Call for init succeed");
                        }
                    }
#ifdef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
                }
                else 
                {
                    if(cypress->haptic_write_status == true)
                    {
                        cypress->haptic_write_status = false;
                        mtouch_info(cypress->log_name, "Ignore the interrupt, as it's for haptic write");
                    }
                    else if(cypress_process_mode_change_interrupt(cypress))
                    {
                        mtouch_info(cypress->log_name, "Mode change interrupt processed, need not process touch");
                    }
#endif
                    else {
                        if (cypress->controller_reset_state == false)
                            rc = cypress_read_touch_data(cypress);

                        else {
                            if (cypress->verbose > 6)
                                mtouch_error (cypress->log_name, "Controller is resetting; unable to process read_touch_data");
                        }

                        if (rc != EOK) {
                            mtouch_error (cypress->log_name, "Error sampling the data from the controller!");
                            error_memory("Cypress_Touch: Error sampling the data from the controller!");
                        }
                    }
#ifdef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
                }
#endif

                if (cypress->polling_rate) {
                    // do nothing as no polling method is used
                }

                if ((BOOTLOADER == cypress->driver_state) && (ACTIVE == cypress->bootloader_state)) {
                    /* We don't want this case */
                } else {
                    /* Platform specific clear irq if exists */
                    if (cypress->platform.tp_clear_irq) {
                        rc = cypress->platform.tp_clear_irq(cypress);
                        if (EOK != rc) {
                            mtouch_error(cypress->log_name, "Error performing platform-specific clear irq");
                            error_memory("Cypress_Touch: Error performing platform-specific clear irq");
                        }
                    }
                }
                rc = pthread_mutex_unlock(&cypress->tp_event_mutex);
                if (rc) {
                    mtouch_error(cypress->log_name, "Can't unlock tp_event_mutex. Err: %d\n", rc);
                }
                InterruptUnmask(cypress->tp_intr, cypress->tp_iid);
                break;
            case CYPRESS_HEARTBEAT_PULSE_CODE:
                /* Timer went off check the heartbeat */
                if (EOK != cypress_operating_mode_heartbeat(cypress)) {
                    mtouch_error (cypress->log_name, "Controller Heartbeat failed.  Resetting the controller and driver");

                    if (EOK != cypress_controller_reset(cypress)) {
                        mtouch_critical (cypress->log_name, "Failed to reset controller");
                    }
                }
                break;
            case CYPRESS_BOOT_WATCHDOG_PULSE_CODE:
                mtouch_info (cypress->log_name, "Received boot watchdog pulse, resetting ...");

                /* Watchdog went off, reboot the controller */
                if (EOK != cypress_controller_reset(cypress)) {
                                        mtouch_critical (cypress->log_name, "Failed to reset controller after bootloader watchdog was triggered");
                                        error_memory("Cypress_Touch: Failed to reset controller after bootloader watchdog was triggered");
                                }
                break;
            case CYPRESS_TP_ERROR_PULSE_CODE:
                if (cypress->verbose > 6)
                    mtouch_info (cypress->log_name, "Error Interrupt");

                switch (cypress_check_error_pin(cypress)) {
                case ERROR_NON_RECOVERABLE:
                    cypress_platform_fini(cypress);
                    return NULL;
                case ERROR_RECOVERABLE:
                    InterruptUnmask(cypress->tp_err_intr, cypress->tp_err_iid);
                    break;
                default:
                    mtouch_critical (cypress->log_name, "Error pin has triggered an interrupt, however we are unable to check the status");
                    error_memory("Cypress_Touch: Error pin has triggered an interrupt, however we are unable to check the status");
                    break;
                }
                break;
            case CYPRESS_MODE_CHG_TIMEOUT_PULSE_CODE:
                mtouch_info (cypress->log_name, "Received mode change timeout pulse");

                rc = cypress_driver_haptic_mode_check(cypress);
                if((rc != EOK) && (cypress->ext_recvid != -1))
                {
                    rc = MsgReply(cypress->ext_recvid, ETIMEDOUT, NULL, 0);
                    if (rc != EOK) {
                        mtouch_info (cypress->log_name, "Failed to send timeout message to application %d", rc);
                        MsgError(cypress->ext_recvid, -errno);
                    }
                    cypress->ext_recvid = -1;
                } else {
                    mtouch_info (cypress->log_name, "Mode change status already sent to application");
                }
                break;
            case CYPRESS_RELEASE_TIMEOUT_EVENT_PULSE_CODE:
            case (CYPRESS_RELEASE_TIMEOUT_EVENT_PULSE_CODE + 1):
            case (CYPRESS_RELEASE_TIMEOUT_EVENT_PULSE_CODE + 2):
            case (CYPRESS_RELEASE_TIMEOUT_EVENT_PULSE_CODE + 3):
                mtouch_warn(cypress->log_name, "Release timer went off for finger: %d! Injecting release", (msg.pulse.code - CYPRESS_RELEASE_TIMEOUT_EVENT_PULSE_CODE));
                /* Modify last report to issue a release for the fingers timer that went off */
                cypress->last_touch_report[msg.pulse.code - CYPRESS_RELEASE_TIMEOUT_EVENT_PULSE_CODE].status = 0;
                mtouch_driver_process_packet(cypress->inputevents_hdl, &cypress->last_touch_report[0], cypress, MTOUCH_PARSER_FLAG_NONE);
                break;
            default:
                mtouch_warn(cypress->log_name, "Received unknown pulse from intr: %d,pid = %d, coid = %d, msglen=%d, ", msg.pulse.code, info.pid, info.coid, info.msglen);
                break;
            }
        } else {
            switch ((int)msg.cmd) {
            case _IO_CONNECT: //name_open() within the client may send this
                MsgReply(rcvid, EOK, NULL, 0);
                break;
            case CYPRESS_CTRL_RESET:
                /* Stop the heartbeat timer */
                if (EOK != cypress_set_heartbeat_timer(cypress, 0)) {
                    mtouch_error (cypress->log_name, "%s failed to set heartbeat timer", __FUNCTION__);
                }

                mtouch_info (cypress->log_name, "Reset requested by control application");

                if (cypress->driver_state == FIRMWARE_UPGRADE) {
                    mtouch_warn (cypress->log_name, "Reset cannot be issued while Firmware Upgrade is in progress");
                    MsgReply(rcvid, -1, NULL, 0);
                    break;
                }

                MsgReply(rcvid, 0, NULL, 0);
                cypress->last_coords.x = 0;
                cypress->last_coords.y = 0;

                mtouch_error (cypress->log_name, "Pulse CTRL_RESET received, resetting ...");
                if (EOK != cypress_controller_reset(cypress)) {
                    MsgReply(rcvid, -1, NULL, 0);
                }

                break;
            case CYPRESS_CTRL_FIRMWARE_UPGRADE:
                /* Stop the heartbeat timer */
                if (EOK != cypress_set_heartbeat_timer(cypress, 0)) {
                    mtouch_error (cypress->log_name, "%s failed to set heartbeat timer", __FUNCTION__);
                }

                mtouch_info (cypress->log_name, "Firmware upgrade requested by control utility");

                cypress->firmware_file = strdup (msg.ctrl.data.buf);
                cypress->firmware_rcvid = rcvid;

                if (cypress->firmware_file == NULL) {
                    mtouch_info (cypress->log_name, "Invalid firmware filename specified");
                    MsgReply(rcvid, -1, NULL, 0);
                    break;
                }

                mtouch_info (cypress->log_name, "Firmware located at %s", cypress->firmware_file);

                /* Attempt to open the firmware file */
                if (EOK != cypress_bootloader_open_file(cypress)) {
                    MsgReply(rcvid, -1, NULL, 0);
                    break;
                }
                mtouch_error (cypress->log_name, "FIRMWARE UPGRADE, resetting ...");
                if (EOK == cypress_controller_reset(cypress)) {
                    cypress->update_firmware = 1;
                    cypress->firmware_update_state = INITIATE_BOOTLOAD;
                    cypress->program_record_state = WRITE;
                }
                break;
            case CYPRESS_CTRL_LASTXY_DATA:
                mtouch_info (cypress->log_name, "Last X/Y coordinates requested by control utility");

                if ((cypress->last_coords.x == 0) || (cypress->last_coords.y == 0)) {
                    MsgReply(rcvid, -1, NULL, 0);
                }

                MsgReply(rcvid, 0, &(cypress->last_coords), sizeof (cypress->last_coords));
                break;
            case CYPRESS_CTRL_IDAC_DATA_SIZE_REQUEST:
                {
/*
                    int data_size = 0;
                    cypress->sensing_mode = msg.ctrl.data.sensing_mode;

                    mtouch_info (cypress->log_name, "IDAC Data size requested by control utility");

                    if (-1 == (data_size = cypress_retrieve_idac_read_length(cypress))) {
                        MsgReply(rcvid, -1, NULL, 0);
                        break;
                    }

                    MsgReply (rcvid, 0, &data_size, sizeof (data_size));
*/
                    /* Don't support this for now */
                    MsgReply(rcvid, -1, NULL, 0);
                }
                break;
            case CYPRESS_CTRL_IDAC_CALIBRATION_REQUIRED:
            case CYPRESS_CTRL_IDAC_CALIBRATION_REQUEST:
                mtouch_info(cypress->log_name, "IDAC calibration requested by control utility");
                cypress->calibration_required = 1;
                break;
            case CYPRESS_CTRL_IDAC_DATA_REQUEST:
                cypress->sensing_mode = msg.ctrl.data.sensing_mode;

                if (cypress->sensing_mode > 7) {
                    mtouch_error (cypress->log_name, "Invalid sensing mode specified via utility");
                    cypress->sensing_mode = 0;
                    MsgReply(rcvid, -1, NULL, 0);
                }

                /* IDAC request issued outside of a firmware update */
                if (CYPRESS_CTRL_IDAC_CALIBRATION_REQUEST == msg.cmd) {
                    cypress_calibrate_idacs(cypress);
                    MsgReply(rcvid, 0, NULL, 0);
                    break;
                }

                /* Request for the IDAC Data was made by the util */
                if (CYPRESS_CTRL_IDAC_DATA_REQUEST == msg.cmd) {
/*
                    uint8_t *reply = NULL;
                    int data_size = cypress_retrieve_idac_data(cypress, &reply);

                    if ((-1 == data_size) || (NULL == reply))
                        MsgReply(rcvid, -1, NULL, 0);

                    MsgReply (rcvid, 0, reply, data_size);
*/
                    /* Don't support this for now */
                    MsgReply(rcvid, -1, NULL, 0);
                    break;
                }

                MsgReply(rcvid, 0, NULL, 0);

                break;
            case CYPRESS_CTRL_CONTROLLER_STATUS_REQUEST:
                {
                    cypress_msg_add_controller_status_t controller_status;

                    if (cypress->verbose > 2)
                        mtouch_info (cypress->log_name, "App is requesting controller status");

                    controller_status.idac_calibration_status = cypress->idac_calibration_status;

                    MsgReply(rcvid, 0, &(controller_status), sizeof (controller_status));
                }
                break;
            case CYPRESS_CTRL_KEYPAD_APP_EVENT_REG:
                mtouch_info (cypress->log_name, "Keypad app is attaching event notification");
                /* Special case were an external app that will inject keys based on the Capsense
                   Buttons pressed on controller */

                cypress->app_rcvid = rcvid;
                cypress->app_notify_event = msg.reg_event.ev;

                if (-1 == MsgReply(rcvid, EOK, NULL, 0)) {
                    perror("MsgReply failed:");
                }

                break;
            case CYPRESS_CTRL_BUTTON_STATUS_REQUEST:
                if (cypress->verbose > 2)
                    mtouch_info (cypress->log_name, "App is requesting button status");

                MsgReply(rcvid, 0, &(cypress->capsense_button_state), sizeof (cypress->capsense_button_state));
                break;
            case CYPRESS_CTRL_CONTROLLER_INFO_REQUEST:
                {
                    cypress_msg_controller_info_t controller_info;

                    if (cypress->verbose > 2)
                        mtouch_info (cypress->log_name, "App is requesting controller info");

                    if (cypress->controller_reset_state == false) {

                        controller_info.product_id = cypress->register_map.cypress_data->ttpid;
                        controller_info.firmware_ver_major = cypress->register_map.cypress_data->fw_ver_major;
                        controller_info.firmware_ver_minor = cypress->register_map.cypress_data->fw_ver_minor;
                        controller_info.silicon_id = cypress->register_map.cypress_data->si_id;

                        MsgReply(rcvid, 0, &(controller_info), sizeof (controller_info));
                    }
                    else {
                        mtouch_error (cypress->log_name, "Controller is resetting; unable to read controller info");
                        MsgError(rcvid, -EAGAIN);
                    }
                }
                break;
            case CYPRESS_CTRL_DESIGN_SIZE_REQUEST:
                if (cypress->register_map.customer_data.dsgn_data) {
                    int data_size = (cypress->register_map.sys_info_mem_map.mdata_ofs - cypress->register_map.sys_info_mem_map.ddata_ofs);

                    MsgReply (rcvid, 0, &data_size, sizeof (data_size));
                } else {
                    MsgReply(rcvid, -1, NULL, 0);
                }
                break;
            case CYPRESS_CTRL_MANUFACTURER_SIZE_REQUEST:
                if (cypress->register_map.customer_data.mfg_data) {
                    int data_size = (cypress->register_map.sys_info_mem_map.map_sz - cypress->register_map.sys_info_mem_map.mdata_ofs);

                    MsgReply (rcvid, 0, &data_size, sizeof (data_size));
                } else {
                    MsgReply(rcvid, -1, NULL, 0);
                }
                break;
            case CYPRESS_CTRL_DESIGN_DATA_REQUEST:
                if (cypress->register_map.customer_data.dsgn_data) {
                    int data_size = (cypress->register_map.sys_info_mem_map.map_sz - cypress->register_map.sys_info_mem_map.mdata_ofs);
                    MsgReply (rcvid, 0, cypress->register_map.customer_data.dsgn_data, data_size);
                } else {
                    MsgReply(rcvid, -1, NULL, 0);
                }
                break;
            case CYPRESS_CTRL_MANUFACTURER_DATA_REQUEST:
                if (cypress->register_map.customer_data.mfg_data) {
                    int data_size = (cypress->register_map.sys_info_mem_map.map_sz - cypress->register_map.sys_info_mem_map.mdata_ofs);
                    MsgReply (rcvid, 0, cypress->register_map.customer_data.mfg_data, data_size);
                } else {
                    MsgReply(rcvid, -1, NULL, 0);
                }
                break;
            case CYPRESS_CTRL_ADDTIONAL_CONTROLLER_INFO_REQUEST:
                {
                    cypress_msg_add_controller_info_t controller_info;

                    if (cypress->verbose > 2)
                        mtouch_info (cypress->log_name, "App is requesting additional controller info");

                    if (cypress->controller_reset_state == false) {

                        controller_info.revctrl = cypress->register_map.cypress_data->revctrl;
                        controller_info.bl_ver_major = cypress->register_map.cypress_data->bl_ver_major;
                        controller_info.bl_ver_minor = cypress->register_map.cypress_data->bl_ver_minor;
                        controller_info.mfgid_sz = cypress->register_map.cypress_data->mfgid_sz;
                        memcpy (controller_info.mfgid, cypress->register_map.cypress_data->mfgid, controller_info.mfgid_sz);
                        controller_info.cyito_ver = cypress->register_map.cypress_data->cyito_ver;
                        controller_info.ttsp_ver_major = cypress->register_map.cypress_data->ttsp_ver_major;
                        controller_info.ttsp_ver_minor = cypress->register_map.cypress_data->ttsp_ver_minor;
                        controller_info.device_info = cypress->register_map.cypress_data->device_info;

                        MsgReply(rcvid, 0, &(controller_info), sizeof (controller_info));
                    }
                    else {
                         mtouch_error (cypress->log_name, "Controller is resetting; unable to read additional controller info");
                         MsgError(rcvid, -EAGAIN);
                    }
                }
                break;
            case CYPRESS_CTRL_NOISE_LEVEL_DATA_REQUEST:
                {
                    cypress_msg_retrieve_noise_data_t noise_data;

                    if (cypress->verbose > 2)
                        mtouch_info (cypress->log_name, "App is requesting noise level data");

                    noise_data.noise_level = 0;
                    noise_data.noise_level_velocity = 0;

                    if (EOK == cypress_operating_mode_fetch_noise_levels(cypress, &(noise_data.noise_level), &(noise_data.noise_level_velocity)))
                        MsgReply(rcvid, 0, &(noise_data), sizeof (noise_data));
                    else
                        MsgReply(rcvid, -1, NULL, 0);
                }
                break;
            case CYPRESS_CTRL_INIT_BASELINES:
                mtouch_info (cypress->log_name, "Init Baselines requested by control application");

                cypress->sensing_mode = msg.ctrl.data.sensing_mode;

                if (cypress->sensing_mode > 7) {
                    mtouch_error (cypress->log_name, "Invalid sensing mode specified via utility");
                    cypress->sensing_mode = 0;
                    MsgReply(rcvid, -1, NULL, 0);
                }

                if (EOK != cypress_init_baselines(cypress))
                    MsgReply(rcvid, -1, NULL, 0);
                else
                    MsgReply(rcvid, 0, NULL, 0);

                break;
            case CYPRESS_CTRL_SAVE_CP_CM_DATA_TO_FILE:
                mtouch_info (cypress->log_name, "Write Cm and Cp Test data to file requested by control application");

                /* Run the Cm and Cp tests and then save the data to a file */
                /* Ensure we are in Config Test Mode */
                if ((old_mode = cypress_get_mode(cypress)) != CONFIGURATION_AND_TEST_MODE) {
                    rc = cypress_set_mode(cypress, CONFIGURATION_AND_TEST_MODE);
                    if (-1 == rc) {
                        mtouch_error (cypress->log_name, "Failed to set controller to config and test mode");
                        error_memory ("Cypress_Touch: Failed to set controller to config and test mode");
                        MsgReply(rcvid, -1, NULL, 0);
                        break;
                    }
                }

                /* Run the tests */
                if (EOK != cypress_run_cp_panel_test(cypress, 1, 1)) {
                    MsgReply(rcvid, -1, NULL, 0);
                    break;
                }

                if (EOK != cypress_run_cm_panel_test(cypress, 1, 1)) {
                    MsgReply(rcvid, -1, NULL, 0);
                    break;
                }

                /* Restore Old Mode if applicable */
                if ((cypress->run_cm_panel_test) && (cypress->run_cp_panel_test)) {
                    if (old_mode != CONFIGURATION_AND_TEST_MODE) {
                        rc = cypress_set_mode(cypress, (old_mode));
                        if (-1 == rc) {
                            mtouch_error (cypress->log_name, "Failed to restore controller mode after config test: %s", __FUNCTION__);
                            error_memory ("Cypress_Touch: Failed to restore controller mode after config test: %s", __FUNCTION__);
                            MsgReply(rcvid, -1, NULL, 0);
                            break;
                        }
                    }
                }

                MsgReply(rcvid, 0, NULL, 0);
                break;
           case CYPRESS_MODE_CHG_REQUEST:
                  mtouch_info (cypress->log_name, "Received touch controller mode change request");
                  rc = cypress_set_mode(cypress, msg.mode_chg.mode);
                  if (rc < 0) {
                     mtouch_error(cypress->log_name, "Error on request set mode rc=%d\n",rc);
                     MsgReply(rcvid, -1, NULL, 0);
                  }
                  cypress->ext_recvid = rcvid;
                  cypress->driver_state = msg.mode_chg.mode;
#ifdef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
                  if(msg.mode_chg.mode == OPERATING_MODE)
                  {
                    cypress->driver_state = OPERATING;
                    cypress->cat_to_operating_mode_state = true;
                  }
#endif
                  /* Start mode change timer, Timeout changed to 200ms */
                  mtouch_info (cypress->log_name, "mode change timer started");
                  if (EOK != cypress_set_mode_change_timer(cypress, cypress->mode_change_sleep)) {
                       mtouch_error (cypress->log_name, "%s failed to set mode change timer", __FUNCTION__);
                  }

                 break;
            default:
                mtouch_error(cypress->log_name, "Received unexpected message %d", msg.cmd);
                MsgError(rcvid, -1);
                break;
            }
        }
    }

    return NULL;
}

static int
set_option(const char* option, const char* value, void* arg)
{
    cypress_dev_t* cypress =(cypress_dev_t *) arg;

    if (0 == strcmp("invert_x", option)) {
        return input_parse_bool(option, value, &cypress->invert_x);
    } else if (0 == strcmp("invert_y", option)) {
        return input_parse_bool(option, value, &cypress->invert_y);
    } else if (0 == strcmp("slave_addr", option)) {
        return input_parse_unsigned(option, value, &cypress->i2c_slave_addr);
    } else if (0 == strcmp("num_touchpts", option)) {
        return input_parse_unsigned(option, value, &cypress->num_touchpts);
    } else if (0 == strcmp("i2c_speed", option)) {
        return input_parse_unsigned(option, value, &cypress->i2c_speed);
    } else if (0 == strcmp("i2c_dev_name", option)) {
        return input_parse_string(option, value, &cypress->i2c_dev_name);
    } else if (0 == strcmp("tp_intr", option)) {
        return input_parse_unsigned(option, value, &cypress->tp_intr);
    } else if (0 == strcmp("width", option)) {
        return input_parse_unsigned(option, value, &cypress->width);
    } else if (0 == strcmp("height", option)) {
        return input_parse_unsigned(option, value, &cypress->height);
    } else if (0 == strcmp("reset_gpio_base_addr", option)) {
        return input_parse_unsigned(option, value, &cypress->reset_gpio_base_addr);
    } else if (0 == strcmp("reset_gpio_pin", option)) {
        return input_parse_unsigned(option, value, &cypress->reset_gpio_pin);
    } else if (0 == strcmp("intr_gpio_pin", option)) {
        return input_parse_unsigned(option, value, &cypress->intr_gpio_pin);
    } else if (0 == strcmp("reset_delay", option)) {
        return input_parse_unsigned(option, value, &cypress->reset_delay);
    } else if (0 == strcmp("reset_recovery_delay", option)) {
        return input_parse_unsigned(option, value, &cypress->reset_recovery_delay);
    } else if (0 == strcmp("operating_mode_reply_retry_delay", option)) {
        return input_parse_unsigned(option, value, &cypress->operating_mode_reply_retry_delay);
    } else if (0 == strcmp("operating_mode_reply_retry_limit", option)) {
        return input_parse_unsigned(option, value, &cypress->operating_mode_reply_retry_limit);
    } else if (0 == strcmp("configuration_reply_retry_delay", option)) {
        return input_parse_unsigned(option, value, &cypress->configuration_reply_retry_delay);
    } else if (0 == strcmp("configuration_reply_retry_limit", option)) {
        return input_parse_unsigned(option, value, &cypress->configuration_reply_retry_limit);
    } else if (0 == strcmp("heartbeat_sleep", option)) {
        return input_parse_unsigned(option, value, &cypress->heartbeat_sleep);
    } else if (0 == strcmp("serializer_slave_addr", option)) {
        return input_parse_unsigned(option, value, &cypress->serializer_slave_addr);
    } else if (0 == strcmp("serializer_i2c_bus", option)) {
        return input_parse_string(option, value, &cypress->serializer_i2c_bus);
    } else if (0 == strcmp("serializer_i2c_speed", option)) {
        return input_parse_unsigned(option, value, &cypress->serializer_i2c_speed);
    } else if (0 == strcmp("verbose", option)) {
        return input_parse_unsigned(option, value, &cypress->verbose);
    } else if (0 == strcmp("boot_watchdog_sleep", option)) {
        return input_parse_unsigned(option, value, &cypress->boot_watchdog_sleep);
    } else if (0 == strcmp("recv_thread_prio", option)) {
        return input_parse_unsigned(option, value, &cypress->recv_thread_prio);
    } else if (0 == strcmp("pulse_prio", option)) {
        return input_parse_unsigned(option, value, &cypress->pulse_prio);
    } else if (0 == strcmp("pps_status_obj", option)) {
        return input_parse_string(option, value, &cypress->pps_status_obj);
    } else if (0 == strcmp("release_timeout", option)) {
        return input_parse_unsigned(option, value, &cypress->release_timeout);
    } else if (0 == strcmp("polling_rate", option)) {
        return input_parse_unsigned(option, value, &cypress->polling_rate);
    } else if (0 == strcmp("soft_reset_recovery_delay", option)) {
        return input_parse_unsigned(option, value, &cypress->soft_reset_recovery_delay);
    } else if (0 == strcmp("run_cm_panel_test", option)) {
        return input_parse_bool(option, value, &cypress->run_cm_panel_test);
    } else if (0 == strcmp("cm_panel_tolerance_max", option)) {
        return input_parse_unsigned(option, value, &cypress->cm_panel_tolerance_max);
    } else if (0 == strcmp("cm_panel_tolerance_min", option)) {
        return input_parse_unsigned(option, value, &cypress->cm_panel_tolerance_min);
    } else if (0 == strcmp("run_cp_panel_test", option)) {
        return input_parse_bool(option, value, &cypress->run_cp_panel_test);
    } else if (0 == strcmp("cp_panel_tolerance_max", option)) {
        return input_parse_unsigned(option, value, &cypress->cp_panel_tolerance_max);
    } else if (0 == strcmp("cp_panel_tolerance_min", option)) {
        return input_parse_unsigned(option, value, &cypress->cp_panel_tolerance_min);
    } else if (0 == strcmp("sensing_mode", option)) {
        return input_parse_unsigned(option, value, (unsigned *) &cypress->sensing_mode);
    } else if (0 == strcmp("save_cm_cp_data", option)) {
        return input_parse_bool(option, value, &cypress->save_cm_cp_data);
    } else if (0 == strcmp("slog_char_limit", option)) {
        return input_parse_unsigned(option, value, &cypress->slog_char_limit);
    } else if (0 == strcmp("tp_err_intr", option)) {
        return input_parse_unsigned(option, value, &cypress->tp_err_intr);
    } else if (0 == strcmp("err_intr_gpio_pin", option)) {
        return input_parse_unsigned(option, value, &cypress->err_intr_gpio_pin);
    } else if (0 == strcmp("detect_panel_detach", option)) {
        return input_parse_bool(option, value, &cypress->detect_panel_detach);
    } else if (0 == strcmp("panel_detach_cp_value", option)) {
        return input_parse_unsigned(option, value, &cypress->panel_detach_cp_value);
    } else if (0 == strcmp("panel_detach_cp_count", option)) {
        return input_parse_unsigned(option, value, &cypress->panel_detach_cp_count);
    } else if (0 == strcmp("panel_detach_cm_value", option)) {
        return input_parse_unsigned(option, value, &cypress->panel_detach_cm_value);
    } else if (0 == strcmp("panel_detach_cm_count", option)) {
        return input_parse_unsigned(option, value, &cypress->panel_detach_cm_count);
    } else if (0 == strcmp("run_calib_cm_fail", option)) {
        return input_parse_unsigned(option, value, &cypress->run_calib_cm_fail);
    } else if (0 == strcmp("run_calib_cp_fail", option)) {
        return input_parse_unsigned(option, value, &cypress->run_calib_cp_fail);
    } else if (0 == strcmp("delay_driver", option)) {
        return input_parse_unsigned(option, value, &cypress->delay_driver);
    } else if (0 == strcmp("pm_dev", option)) {
        return input_parse_string(option, value, &cypress->pm_dev);
    } else if (0 == strcmp("hard_reset", option)) {
        return input_parse_unsigned(option, value, &cypress->hard_reset);
    } else if (0 == strcmp("win_height", option)) {
        return input_parse_unsigned(option, value, &cypress->win_height);
    } else if (0 == strcmp("win_width", option)) {
        return input_parse_unsigned(option, value, &cypress->win_width);
    } else if (0 == strcmp("disp_id", option)) {
        return input_parse_unsigned(option, value, &cypress->disp_id);
    } else if (0 == strcmp("win_xpos", option)) {
        return input_parse_unsigned(option, value, &cypress->win_xpos);
    } else if (0 == strcmp("win_ypos", option)) {
        return input_parse_unsigned(option, value, &cypress->win_ypos);
    } else if (0 == strcmp("win_name", option)) {
        return input_parse_string(option, value, &cypress->win_name);
    } else if (0 == strcmp("dev_ctrl_path", option)) {
        return input_parse_string(option, value, &cypress->dev_ctrl_path);
    } else if (0 == strcmp("dev_status_path", option)) {
        return input_parse_string(option, value, &cypress->dev_status_path);
    } else if (0 == strcmp("attach_point", option)) {
        return input_parse_string(option, value, &cypress->attach_point);
    } else if (0 == strcmp("fidm_attach_point", option)) {
        return input_parse_string(option, value, &cypress->fidm_attach_point);
    } else if (0 == strcmp("sync_point", option)) {
        return input_parse_unsigned(option, value, &cypress->sync_point);
    } else if (0 == strcmp("play_haptic_config", option)) {
        return input_parse_bool(option, value, (unsigned int*)&cypress->play_haptic_config);
    } else {
        mtouch_error(cypress->log_name, "Invalid option: '%s'", option);
        return -1;
    }

    return -1;
}


void*
mtouch_driver_init(const char* options)
{
    int retval =0;
    int fd = -1;
    char str1[20];
    pthread_attr_t pattr;
    sched_param_t param;
    pthread_attr_t pattr_msg;
    sched_param_t param_msg;
    BootctrlMode mode;
    cypress_dev_t* cypress =(cypress_dev_t *) calloc(1, sizeof(*cypress));

    if (NULL == cypress) {
        mtouch_error("cypress", "Failed to alloc device memory");
        return NULL;
    }

    cypress->log_name = "cypress";

    mtouch_info (cypress->log_name, "Cypress driver init");

    /* Initialize defaults */
    cypress->touch_reg_count = 8;
    cypress->delay_driver = 0;

    /* These are the touchscreen max x and max y values */
    cypress->width = 1024;
    cypress->height = 1024;
    cypress->invert_y = 0;
    cypress->invert_x = 0;
    cypress->i2c_low_clock_tweak = 0;
    cypress->i2c_high_clock_tweak = 0;
    cypress->tp_iid = -1;
    cypress->tp_err_iid = -1;
    cypress->recv_thread_prio = CYPRESS_TP_THREAD_PRIORITY;
    cypress->pulse_prio = CYPRESS_TP_PULSE_PRIORITY;
    cypress->attach = NULL;
    cypress->coid = -1;
    cypress->power_state = CYPRESS_POWER_STATE_NONE;
    cypress->debug_port = 0;
    cypress->debug_priority = 20;
    cypress->debug_retry_count = 36;
    cypress->debug_retry_interval = 5000;
    cypress->debug_raw_mode = 0;
    cypress->seq_id = 0;
    cypress->protocol = 0;
    cypress->i2c_lib = NULL;
    cypress->i2c_fd = -1;
    cypress->i2c_slave_addr = CYPRESS_I2C_SLAVE_ADDR;
    cypress->num_touchpts = CYPRESS_DEFAULT_TOUCHPOINTS;
    cypress->i2c_dev_name = NULL;
    cypress->tp_intr = 0;
    cypress->tp_err_intr = 0;
    cypress->reset_gpio_base_addr = 0;
    cypress->reset_gpio_pin = 0;
    cypress->intr_gpio_pin = 0;
    cypress->err_intr_gpio_pin = 0;
    cypress->reset_delay = 500;
    cypress->reset_recovery_delay = 400000;
    cypress->driver_state = UNKNOWN;
    cypress->bootloader_state = IDLE;
    cypress->controller_reset_state = true;
    cypress->verbose = 0;
    cypress->firmware_file = NULL;
    cypress->last_coords.x = 0;
    cypress->last_coords.y = 0;
    cypress->soft_reset_recovery_delay = 200;

    /* Platform specific data */
    cypress->platform.tp_init = NULL;
    cypress->platform.tp_reset = NULL;
    cypress->platform.tp_fini = NULL;
    cypress->platform.tp_clear_irq = NULL;
    cypress->platform.data = NULL;

    /* Firmware */
    cypress->update_firmware = 0;
    cypress->chip_revision = 0;
    cypress->file_chip_revision = 0;
    cypress->silicon_id = 0;
    cypress->file_silicon_id = 0;
    cypress->firmware_update_state = OPEN_FILE;
    cypress->program_record_state = WRITE;
    cypress->retry_limit = MAX_RETRY_LIMIT;
    cypress->retry_attempts = 0;
    cypress->sensing_mode = 0;
    cypress->calibration_required = 0;
    cypress->panel_detach_detected = 0;
    cypress->firmware_lines_complete = 1;
    cypress->firmware_line_count = 0;
    cypress->idac_calibration_status = 0; /* -1 = Fail, 0 = Not Run, 1 = Success */

    /* Configuration and Test Mode Parameters */
    cypress->configuration_reply_retry_limit = CYPRESS_CONFIGURATION_MODE_MAX_REPLY_ATTEMPTS;
    cypress->configuration_reply_retry_delay = CYPRESS_CONFIGURATION_MODE_REPLY_DEFAULT_DELAY;
    cypress->operating_mode_reply_retry_limit = CYPRESS_OPERATING_MODE_MAX_REPLY_ATTEMPTS;
    cypress->operating_mode_reply_retry_delay = CYPRESS_OPERATING_MODE_REPLY_DEFAULT_DELAY;

    /* Heartbeat */
    cypress->heartbeat_sleep = CYPRESS_DEFAULT_HEARTBEAT_SLEEP;

    /* Bootloader watchdog*/
    cypress->boot_watchdog_sleep = 3;

    /* mode change timeout */
    cypress->mode_change_sleep = CYPRESS_MODE_SWITCH_TIMEOUT_NANOSEC;
    cypress->ext_recvid = -1;

    /* Button */
    cypress->num_button_regs = 0;

    /* Serializer */
    cypress->serializer_slave_addr = 0;
    cypress->serializer_i2c_bus = 0;
    cypress->serializer_i2c_speed = 0;

    /* PPS Logging */
    cypress->pps_status_obj = NULL;
    cypress->pps_status_fd = -1;

    cypress->slog_char_limit = CYPRESS_DEFAULT_SLOG_CHAR_LIMIT;

    /* Release timeout */
    cypress->release_timeout = 0;

    /* Polling (Debug) */
    cypress->polling_rate = 0;

    /* Manufacturer Tests */
    /* Cm Panel Test */
    cypress->run_cm_panel_test = 0;
    cypress->cm_panel_ref_max = NULL;
    cypress->cm_panel_ref_min = NULL;
    cypress->cm_panel_tolerance_max = 0;
    cypress->cm_panel_tolerance_min = 0;
    cypress->run_calib_cm_fail = 0;
    /* Cp Panel Test */
    cypress->run_cp_panel_test = 0;
    cypress->cp_panel_tx_ref_max = NULL;
    cypress->cp_panel_tx_ref_min = NULL;
    cypress->cp_panel_rx_ref_max = NULL;
    cypress->cp_panel_rx_ref_min = NULL;
    cypress->cp_panel_tolerance_max = 0;
    cypress->cp_panel_tolerance_min = 0;
    cypress->run_calib_cp_fail = 0;

    /* Save Cm & Cp Panel Test Data */
    cypress->save_cm_cp_data = 0;
    /* Detect Panel Detach */
    cypress->detect_panel_detach = 0;
    cypress->panel_detach_cp_value = 25000;
    cypress->panel_detach_cp_count = 10;
    cypress->panel_detach_cm_value = 50;
    cypress->panel_detach_cm_count = 156;

    /* Config & Test Details */
    cypress->config_block_size = 0;

    /* Error pin */
    cypress->error_pin_status = 0;

    /* default dev pm */
    cypress->pm_dev = DEFAULT_PM_DEV_NODE;

    /* window default to bosch display size*/
    cypress->win_height = 720;
    cypress->win_width = 1280;
    cypress->win_xpos = 0;
    cypress->win_ypos = 0;
    cypress->disp_id = 2;
    cypress->win_name = "qvm_window";
    cypress->dev_ctrl_path = "/dev/vcd/display-binder/control";
    cypress->dev_status_path = "/dev/vcd/display-binder/status";
    cypress->display_group_id = -1;

    cypress->attach_point = NULL;
    cypress->fidm_attach_point = NULL;

    /* hard reset*/
    cypress->hard_reset = 1;
    cypress->rtd_readflag = 0;
    cypress->intr_coid = -1;
    cypress->sync_point = 1;
    cypress->repeatCount = 0;
    cypress->cy_str_state = PM_STATE_COMPLETE;
    cypress->play_haptic_config = false;
#ifdef BOSCH_RTC_2487095_HID_CYPRESS_DATASHEET_ALIGNMENT_CHANGES
    cypress->sysinfo_to_operating_mode_state = false;
    cypress->cat_to_operating_mode_state = false;
    cypress->haptic_write_status = false;
#endif

    cypress->retry_init_attempts = 0;
    memset(cypress->num_prev_recs, 0, sizeof(cypress->num_prev_recs));  // Initialize num_prev_recs to 0
    cypress->tp_event_mutex = (pthread_mutex_t)PTHREAD_MUTEX_INITIALIZER;
    cypress->ctrl_reset_mutex = (pthread_mutex_t)PTHREAD_MUTEX_INITIALIZER;

    if (InstallLocalErrMemReader(RINGBUFFER_SIZE) != 0)
    {
        mtouch_error (cypress->log_name, "errmem low priority worker thread creation failed !!!");
    }

    input_parseopts(options, set_option, cypress);

    sprintf(str1, "cypress_disp_id_%d", cypress->disp_id);
    cypress->log_name = strdup(str1);

    mode = pal_BootControl_GetActiveBootMode();
    mtouch_debug (cypress->log_name, "Boot mode is %d", mode);
    if(BOOT_CONTROL_RECOVERY_MODE == mode)
    {
        mtouch_error (cypress->log_name, "System in recovery mode, retry limit = 0");
        cypress->retry_limit = 0;
    }
    else{
        mtouch_debug (cypress->log_name, "System in normal mode");
    }

    /* Open PPS object to post status */
    if (NULL != cypress->pps_status_obj) {
        cypress->pps_status_fd = open (cypress->pps_status_obj, O_WRONLY | O_CREAT, 0666);
        if (-1 == cypress->pps_status_fd) {
            mtouch_error (cypress->log_name, "Failed to open PPS object %s", cypress->pps_status_obj);
            error_memory("Cypress_Touch: Failed to open PPS object %s", cypress->pps_status_obj);
            return NULL;
        }

        cypress_update_pps_status(cypress, "status::bootloader");
    }

    if ((cypress->run_cm_panel_test) || (cypress->save_cm_cp_data))
        if (!cypress->sensing_mode) {
            mtouch_error (cypress->log_name, "sensing_mode must be set and cannot be zero!");
            return NULL;
        }

    if ((cypress->run_cp_panel_test) || (cypress->save_cm_cp_data))
        if (!cypress->sensing_mode) {
            mtouch_error (cypress->log_name, "sensing_mode must be set and cannot be zero!");
            return NULL;
        }

    /* Tolerance of minimum reference data for Cm and Cp panel test is a percentage value.
       If this value is greater than 100, then the minimum reference data will be negative.
       Therefore, the limit of this value should be set to 100
     */
    if (cypress->cm_panel_tolerance_min > 100)
        cypress->cm_panel_tolerance_min = 100;

    if (cypress->cp_panel_tolerance_min > 100)
        cypress->cp_panel_tolerance_min = 100;

    if (cypress->release_timeout)
        cypress->release_timeout *= 1000000;

    if ((cypress->boot_watchdog_sleep) && (cypress->boot_watchdog_sleep < CYPRESS_MIN_BOOT_WATCHDOG_SLEEP)) {
        mtouch_warn(cypress->log_name, "!!!!Boot watchdog sleep is too short!!!! Firmware upgrade will fail.  Setting to default (%d secs)", CYPRESS_MIN_BOOT_WATCHDOG_SLEEP);
        cypress->boot_watchdog_sleep = CYPRESS_MIN_BOOT_WATCHDOG_SLEEP;
    }

    if (cypress->polling_rate) {
        mtouch_info(cypress->log_name, "Using polling mode, rate: %d", cypress->polling_rate);
    }

    if (cypress->num_touchpts > CYPRESS_MAX_TOUCHPOINTS) {
        mtouch_error(cypress->log_name, "num_touchpts exceeds limit of %d", CYPRESS_MAX_TOUCHPOINTS);
        cypress->num_touchpts = CYPRESS_MAX_TOUCHPOINTS;
    }

    if (cypress->attach_point == NULL) {
        cypress->attach_point = strdup (ATTACH_POINT);
    }

    if (cypress->fidm_attach_point == NULL) {
        cypress->fidm_attach_point = strdup (FIDM_TOUCH_ATTACH_POINT);
    }

    /* Create channel for interrupt and cypress_ctrl messages */
    if ((cypress->attach = name_attach(NULL, cypress->attach_point, 0)) == NULL) {
        mtouch_error(cypress->log_name, "%s: name_attach: %s", __FUNCTION__, strerror(errno));
        error_memory("Cypress_Touch: %s: name_attach: %s", __FUNCTION__, strerror(errno));
        goto fail2;
    }

    cypress->coid = ConnectAttach(0, 0, cypress->attach->chid, _NTO_SIDE_CHANNEL, 0);
    if (-1 == cypress->coid) {
        mtouch_error(cypress->log_name, "%s: ConnectAttach: %s", __FUNCTION__, strerror(errno));
        error_memory("Cypress_Touch: %s: ConnectAttach: %s", __FUNCTION__, strerror(errno));
        goto fail3;
    }

    /* Create channel for external messages */
    if ((cypress->fidm_attach = name_attach(NULL, cypress->fidm_attach_point, 0)) == NULL) {
            mtouch_error(cypress->log_name, "%s: name_attach: %s", __FUNCTION__, strerror(errno));
            error_memory("Cypress_Touch: %s: name_attach: %s", __FUNCTION__, strerror(errno));
            goto fail4;
    }

    cypress->fidm_coid = ConnectAttach(0, 0, cypress->fidm_attach->chid, _NTO_SIDE_CHANNEL, 0);
    if (-1 == cypress->fidm_coid) {
            mtouch_error(cypress->log_name, "%s: ConnectAttach: %s", __FUNCTION__, strerror(errno));
            error_memory("Cypress_Touch: %s: ConnectAttach: %s", __FUNCTION__, strerror(errno));
            goto fail5;
    }

    cypress->intr_coid = cypress->coid;

    if (NULL == cypress->i2c_lib) {
        cypress->i2c_lib = strdup("qc-i2c-client");
    }

    if (EOK != get_qc_libi2c_funcs(cypress->i2c_lib, &cypress->i2c_funcs)) {
        mtouch_error(cypress->log_name, "%s: unrecognized i2c_lib: %s", __FUNCTION__, cypress->i2c_lib);
        error_memory ("Cypress_Touch: %s: unrecognized i2c_lib: %s", __FUNCTION__, cypress->i2c_lib);
        goto fail6;
    }

    /* Attach to i2c */
    if ((cypress->i2c_fd = cypress_i2c_open(cypress->i2c_dev_name, cypress->i2c_slave_addr, cypress->i2c_speed)) < 0) {
        mtouch_error(cypress->log_name, "%s: failed to init i2c_lib: %s", __FUNCTION__, cypress->i2c_lib);
        error_memory("Cypress_Touch: %s: failed to init i2c_lib: %s", __FUNCTION__, cypress->i2c_lib);
        goto fail6;
    } else {
        mtouch_info(cypress->log_name, "%s: registration with %s successful", __FUNCTION__, cypress->i2c_dev_name);
    }

    if (EOK != cypress_register_pm(cypress)){
        mtouch_error(cypress->log_name, "%s: registration with %s failed", __FUNCTION__, cypress->pm_dev );
        error_memory("Cypress_Touch: %s: registration with %s failed", __FUNCTION__, cypress->pm_dev );}
    else
        mtouch_info(cypress->log_name, "%s: registration with %s successful", __FUNCTION__, cypress->pm_dev );

    /* Attach to libinputevents */
    if (EOK != attach_driver(cypress)) {
        goto fail7;
    }

    if (cypress->intr_gpio_pin) {
        if (-1 == (tp_intr_gpio_to_irq(cypress))) {
            goto fail8;
        }
    }

    /* Attach the TP interrupt */
    if (-1 == (cypress->tp_iid = tp_intr_attach(cypress, &cypress->tp_intrevent, cypress->tp_intr, CYPRESS_TP_PULSE_CODE))) {
        goto fail8;
    }

    /* create a window to forward events to guest OS */
    if (0 > cypress_create_qvm_window(cypress)) {
        mtouch_error(cypress->log_name, "Failed to create window for qvm");
        goto fail8;
    }

    /* Attach the Error Interrupt (if set) */
    if (cypress->tp_err_intr) {
        if (-1 == (cypress->tp_err_iid = tp_intr_attach(cypress, &cypress->tp_err_intrevent, cypress->tp_err_intr, CYPRESS_TP_ERROR_PULSE_CODE))) {
            goto fail9;
        }
    }
    /* Display Group ID */
    retval = cypress_get_display_grpid(cypress);
    if(retval != 0)
    {
        mtouch_error(cypress->log_name, "Failed to fetch Display Group Id: %d in init:retval: %d", cypress->display_group_id, retval);
    }

    if (EOK != pthread_mutex_init(&cypress->tp_event_mutex, NULL)) {
        mtouch_error(cypress->log_name, "Failed to pthread_mutex_init tp_event_mutex");
    }

    if (EOK != pthread_mutex_init(&cypress->ctrl_reset_mutex, NULL)) {
        mtouch_error(cypress->log_name, "Failed to pthread_mutex_init ctrl_reset_mutex");
    }
    pthread_attr_init(&pattr_msg);
    pthread_getschedparam(pthread_self(), NULL, &param_msg);
    param_msg.sched_priority = cypress->recv_thread_prio;
    pthread_attr_setschedparam(&pattr_msg, &param_msg);
    pthread_attr_setinheritsched(&pattr_msg, PTHREAD_EXPLICIT_SCHED);

    if (EOK != pthread_create(&cypress->fidm_thread, &pattr_msg, cypress_ext_msg_handler, cypress)) {
        mtouch_error(cypress->log_name, "Failed to create the intr thread");
        error_memory("Cypress_Touch: Failed to create the intr thread");
        goto fail9;
    }

    pthread_attr_init(&pattr);
    pthread_getschedparam(pthread_self(), NULL, &param);
    param.sched_priority = cypress->recv_thread_prio;
    pthread_attr_setschedparam(&pattr, &param);
    pthread_attr_setinheritsched(&pattr, PTHREAD_EXPLICIT_SCHED);

    if (EOK != pthread_create(&cypress->recv_thread, &pattr, tp_recv_thread, cypress)) {
        mtouch_error(cypress->log_name, "Failed to create the intr thread %d", errno);
        error_memory ("Cypress_Touch: Failed to create the intr thread");
        goto fail10;
    }

    pthread_setname_np(cypress->recv_thread, "mtouch-cypress-isr");
    pthread_setname_np(cypress->fidm_thread, "mtouch-cypress_msg_handler");

    if (cypress->sync_point) {
    /* Create file to signal that the driver is initialized and ready to serve clients */
       fd = (open("/tmp/sync/mtouch_initialized", O_WRONLY | O_CREAT | O_TRUNC,
                     S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH));
       if (fd == -1){
           mtouch_error(cypress->log_name, "%s failed to create sync file", __FUNCTION__);
       }
    } else {
       mtouch_info(cypress->log_name, "diagnostics sync point not enabled %d", cypress->sync_point);
    }

    if (fd != -1)
        close(fd);

    return cypress;

fail10:
    pthread_cancel(cypress->fidm_thread);
    pthread_join(cypress->fidm_thread, NULL);

fail9:
    if (cypress->tp_err_intr)
        InterruptDetach(cypress->tp_err_iid);

    InterruptDetach(cypress->tp_iid);
    cypress->tp_iid = -1;
    pthread_mutex_destroy(&cypress->tp_event_mutex);
    pthread_mutex_destroy(&cypress->ctrl_reset_mutex);

fail8:
    mtouch_driver_detach(cypress->inputevents_hdl);
    cypress->inputevents_hdl = NULL;

fail7:
    cypress_i2c_close(cypress->i2c_fd);

fail6:
    cypress_platform_fini(cypress);

    if (cypress->attach && cypress->attach->chid >= 0) {
        ChannelDestroy(cypress->attach->chid);
        cypress->attach->chid = -1;
    }

    if (cypress->fidm_attach && cypress->fidm_attach->chid >= 0) {
        ChannelDestroy(cypress->fidm_attach->chid);
        cypress->fidm_attach->chid = -1;
    }

fail5:
    name_detach(cypress->fidm_attach, 0);
    cypress->fidm_attach = NULL;

fail4:
    ConnectDetach(cypress->coid);
    cypress->coid = -1;

fail3:
    //if (cypress->attach && cypress->attach->chid >= 0) {
     name_detach(cypress->attach, 0);
     cypress->attach = NULL;
    //}

fail2:
    free(cypress);
    return NULL;
}

void
mtouch_driver_fini(void* dev)
{
    cypress_dev_t* cypress =(cypress_dev_t *) dev;
    int i;

    if (cypress->debug_port) {
        pthread_cancel(cypress->debug_thread);
        pthread_join(cypress->debug_thread, NULL);
    }

    pthread_mutex_destroy(&cypress->tp_event_mutex);
    pthread_mutex_destroy(&cypress->ctrl_reset_mutex);

    pthread_cancel(cypress->recv_thread);
    pthread_join(cypress->recv_thread, NULL);
    pthread_cancel(cypress->fidm_thread);
    pthread_join(cypress->fidm_thread, NULL);

    cypress_platform_fini(cypress);

    if (cypress->tp_iid >= 0) {
        InterruptDetach(cypress->tp_iid);
        cypress->tp_iid = -1;
    }

    if (cypress->tp_err_intr) {
        if (cypress->tp_err_iid >= 0) {
            InterruptDetach(cypress->tp_err_iid);
            cypress->tp_err_iid = -1;
        }
    }

    if (cypress->inputevents_hdl) {
        mtouch_driver_detach(cypress->inputevents_hdl);
        cypress->inputevents_hdl = NULL;
    }

    if (cypress->fidm_attach && cypress->fidm_attach->chid >= 0) {
        ChannelDestroy(cypress->fidm_attach->chid);
        cypress->fidm_attach->chid = -1;
        ConnectDetach(cypress->fidm_coid);
        cypress->fidm_coid = -1;
        name_detach(cypress->fidm_attach, 0);
        cypress->fidm_attach = NULL;
    }

    if ((cypress->attach) && (cypress->attach->chid >= 0)) {
        ChannelDestroy(cypress->attach->chid);
        cypress->attach->chid = -1;
        ConnectDetach(cypress->coid);
        cypress->coid = -1;
        name_detach(cypress->attach, 0);
        cypress->attach = NULL;
    }

    close(cypress->pm_fd);

    timer_delete(cypress->heartbeat_timerid);

    timer_delete(cypress->boot_watchdog_timerid);

    timer_delete(cypress->mode_change_timerid);

    if (cypress->release_timeout) {
        for (i = 0; i < cypress->num_touchpts; i++)
            timer_delete(cypress->inject_release_timerid[i]);

        free (cypress->inject_release_timerid);
    }

    if (cypress->last_touch_report)
        free (cypress->last_touch_report);

    cypress_i2c_close(cypress->i2c_fd);

    system_information_clean_up(cypress);
    free(cypress);
}
#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn.ott.qnx.com/product/branches/7.0.0/trunk/hardware/mtouch/cypress/touch.c $ $Rev: 888004 $")
#endif
